<?php

use App\Models\Setting;
use Illuminate\Foundation\Inspiring;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Schedule;

Artisan::command('inspire', function () {
    $this->comment(Inspiring::quote());
})->purpose('Display an inspiring quote');

// Get timezone from config (which is loaded from settings)
$timezone = config('app.timezone', 'America/Santo_Domingo');

// Enviar recordatorios de pago diariamente a las 9:00 AM
// Solo se ejecuta si cron_reminders_enabled está activo
Schedule::command('payments:send-reminders')
    ->dailyAt('09:00')
    ->timezone($timezone)
    ->onOneServer()
    ->withoutOverlapping()
    ->when(function () {
        return (bool) Setting::get('cron_reminders_enabled', false);
    });

// Verificar suscripciones próximas a vencer (7 días)
// Solo se ejecuta si cron_billing_enabled está activo
Schedule::command('subscriptions:check-expiring --days=7')
    ->dailyAt('08:00')
    ->timezone($timezone)
    ->onOneServer()
    ->when(function () {
        return (bool) Setting::get('cron_billing_enabled', false);
    });

// Verificar facturas vencidas (más de 7 días sin pagar)
// Solo se ejecuta si cron_billing_enabled está activo
Schedule::command('invoices:check-overdue --days=7')
    ->dailyAt('10:00')
    ->timezone($timezone)
    ->onOneServer()
    ->when(function () {
        return (bool) Setting::get('cron_billing_enabled', false);
    });

// Procesar expiración de planes de coach
// Solo se ejecuta si cron_billing_enabled está activo
Schedule::command('coach-plans:process-expirations')
    ->dailyAt('07:00')
    ->timezone($timezone)
    ->onOneServer()
    ->when(function () {
        return (bool) Setting::get('cron_billing_enabled', false);
    });

// Actualizar suscripciones expiradas automáticamente
// Marca como 'expired' las suscripciones activas que pasaron su end_date
Schedule::command('subscriptions:update-expired')
    ->dailyAt('00:01')
    ->timezone($timezone)
    ->onOneServer()
    ->withoutOverlapping();


// Procesar cola de trabajos cada minuto (siempre activo)
Schedule::command('queue:work --stop-when-empty')
    ->everyMinute()
    ->withoutOverlapping()
    ->runInBackground();

// Reintentar correos fallidos cada 15 minutos
Schedule::command('emails:retry-failed --max-attempts=3 --limit=20')
    ->everyFifteenMinutes()
    ->timezone($timezone)
    ->onOneServer()
    ->withoutOverlapping();

// Limpiar trabajos fallidos antiguos (más de 30 días) - mantenimiento del sistema
Schedule::command('queue:prune-failed --hours=720')
    ->weekly()
    ->timezone($timezone);

// Limpiar cache de vistas y configuración (semanal) - mantenimiento del sistema
Schedule::command('view:clear')
    ->weekly()
    ->timezone($timezone);

// Limpiar logs antiguos (mensual) - mantiene últimos 14 días
Schedule::command('log:clear --keep=14')
    ->monthly()
    ->timezone($timezone);

// Limpiar logs de correo antiguos (semanal) - mantiene últimos 30 días
Schedule::command('model:prune', ['--model' => 'App\\Models\\EmailLog'])
    ->weekly()
    ->timezone($timezone);
