<?php

namespace Database\Seeders;

use App\Models\ExerciseLibrary;
use Illuminate\Database\Seeder;

class ExerciseLibrarySeeder extends Seeder
{
    /**
     * Seed the exercise library with default exercises
     * GIFs sourced from: https://github.com/yuhonas/free-exercise-db (MIT License)
     * Alternative sources:
     * - Wger API: https://wger.de/api/v2/exerciseimage/
     * - MuscleWiki: https://musclewiki.com (for reference)
     */
    public function run(): void
    {
        $exercises = [
            // ================= PECHO =================
            [
                'name' => 'Press de Banca',
                'description' => 'Ejercicio fundamental para el desarrollo del pecho, hombros y tríceps.',
                'category' => 'fuerza',
                'muscle_group' => 'pecho',
                'secondary_muscles' => ['triceps', 'hombros'],
                'difficulty' => 'intermedio',
                'is_compound' => true,
                'equipment' => 'barra',
                'instructions' => '1. Acuéstate en el banco con los pies firmemente en el suelo.
2. Agarra la barra con un agarre ligeramente más ancho que los hombros.
3. Baja la barra controladamente hasta que toque el pecho.
4. Empuja la barra hacia arriba hasta extender los brazos.
5. Mantén los omóplatos retraídos durante todo el movimiento.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Barbell_Bench_Press/0.gif',
            ],
            [
                'name' => 'Press de Banca Inclinado',
                'description' => 'Variación que enfatiza la parte superior del pecho.',
                'category' => 'fuerza',
                'muscle_group' => 'pecho',
                'secondary_muscles' => ['hombros', 'triceps'],
                'difficulty' => 'intermedio',
                'is_compound' => true,
                'equipment' => 'barra',
                'instructions' => '1. Ajusta el banco a 30-45 grados de inclinación.
2. Acuéstate y agarra la barra con agarre medio.
3. Baja la barra hacia la parte superior del pecho.
4. Empuja hacia arriba hasta extender los brazos.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Barbell_Incline_Bench_Press/0.gif',
            ],
            [
                'name' => 'Flexiones de Pecho',
                'description' => 'Ejercicio básico de peso corporal para fortalecer pecho, hombros y tríceps.',
                'category' => 'calistenia',
                'muscle_group' => 'pecho',
                'secondary_muscles' => ['triceps', 'hombros', 'abdominales'],
                'difficulty' => 'principiante',
                'is_compound' => true,
                'equipment' => 'ninguno',
                'instructions' => '1. Colócate en posición de plancha con las manos a la altura de los hombros.
2. Mantén el cuerpo recto como una tabla.
3. Baja el cuerpo flexionando los codos hasta que el pecho casi toque el suelo.
4. Empuja hacia arriba hasta la posición inicial.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Push-Up/0.gif',
            ],
            [
                'name' => 'Aperturas con Mancuernas',
                'description' => 'Ejercicio de aislamiento para estirar y contraer el pecho.',
                'category' => 'fuerza',
                'muscle_group' => 'pecho',
                'secondary_muscles' => ['hombros'],
                'difficulty' => 'intermedio',
                'is_compound' => false,
                'equipment' => 'mancuernas',
                'instructions' => '1. Acuéstate en un banco plano con una mancuerna en cada mano.
2. Extiende los brazos sobre el pecho con una ligera flexión de codos.
3. Baja los brazos en arco hacia los lados.
4. Contrae el pecho para volver a la posición inicial.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Dumbbell_Flyes/0.gif',
            ],
            [
                'name' => 'Press con Mancuernas',
                'description' => 'Versión con mancuernas del press de banca para mayor rango de movimiento.',
                'category' => 'fuerza',
                'muscle_group' => 'pecho',
                'secondary_muscles' => ['triceps', 'hombros'],
                'difficulty' => 'intermedio',
                'is_compound' => true,
                'equipment' => 'mancuernas',
                'instructions' => '1. Acuéstate en el banco con una mancuerna en cada mano.
2. Posiciona las mancuernas a los lados del pecho.
3. Empuja hacia arriba extendiendo los brazos.
4. Baja controladamente hasta la posición inicial.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Dumbbell_Bench_Press/0.gif',
            ],
            [
                'name' => 'Fondos en Paralelas',
                'description' => 'Ejercicio compuesto que trabaja pecho inferior, tríceps y hombros.',
                'category' => 'calistenia',
                'muscle_group' => 'pecho',
                'secondary_muscles' => ['triceps', 'hombros'],
                'difficulty' => 'intermedio',
                'is_compound' => true,
                'equipment' => 'barra_dominadas',
                'instructions' => '1. Agarra las barras paralelas y eleva tu cuerpo.
2. Inclina el torso ligeramente hacia adelante para enfatizar el pecho.
3. Baja el cuerpo flexionando los codos hasta 90 grados.
4. Empuja hacia arriba hasta extender los brazos.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Chest_Dips/0.gif',
            ],
            
            // ================= ESPALDA =================
            [
                'name' => 'Dominadas',
                'description' => 'Ejercicio rey para el desarrollo de la espalda y bíceps.',
                'category' => 'calistenia',
                'muscle_group' => 'espalda',
                'secondary_muscles' => ['biceps', 'antebrazos'],
                'difficulty' => 'intermedio',
                'is_compound' => true,
                'equipment' => 'barra_dominadas',
                'instructions' => '1. Agarra la barra con las palmas hacia adelante, más ancho que los hombros.
2. Cuelga con los brazos completamente extendidos.
3. Tira del cuerpo hacia arriba hasta que el mentón supere la barra.
4. Baja controladamente hasta la posición inicial.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Pull-Up/0.gif',
            ],
            [
                'name' => 'Remo con Barra',
                'description' => 'Ejercicio compuesto para desarrollar grosor en la espalda.',
                'category' => 'fuerza',
                'muscle_group' => 'espalda',
                'secondary_muscles' => ['biceps', 'trapecio'],
                'difficulty' => 'intermedio',
                'is_compound' => true,
                'equipment' => 'barra',
                'instructions' => '1. Inclínate hacia adelante con la espalda recta.
2. Agarra la barra con agarre prono.
3. Tira de la barra hacia el abdomen.
4. Baja controladamente manteniendo la tensión.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Barbell_Bent_Over_Row/0.gif',
            ],
            [
                'name' => 'Remo con Mancuerna',
                'description' => 'Remo unilateral para corregir desequilibrios musculares.',
                'category' => 'fuerza',
                'muscle_group' => 'espalda',
                'secondary_muscles' => ['biceps'],
                'difficulty' => 'principiante',
                'is_compound' => true,
                'equipment' => 'mancuernas',
                'instructions' => '1. Coloca una rodilla y mano en el banco.
2. Agarra la mancuerna con la otra mano.
3. Tira de la mancuerna hacia la cadera.
4. Baja controladamente y repite.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Dumbbell_Row/0.gif',
            ],
            [
                'name' => 'Jalón al Pecho',
                'description' => 'Ejercicio de polea para desarrollar la anchura de la espalda.',
                'category' => 'fuerza',
                'muscle_group' => 'espalda',
                'secondary_muscles' => ['biceps'],
                'difficulty' => 'principiante',
                'is_compound' => true,
                'equipment' => 'cable',
                'instructions' => '1. Siéntate en la máquina y agarra la barra ancha.
2. Inclina ligeramente el torso hacia atrás.
3. Tira de la barra hacia el pecho.
4. Controla el movimiento de vuelta.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Lat_Pulldown/0.gif',
            ],
            [
                'name' => 'Peso Muerto',
                'description' => 'Ejercicio fundamental para toda la cadena posterior.',
                'category' => 'fuerza',
                'muscle_group' => 'espalda',
                'secondary_muscles' => ['gluteos', 'isquiotibiales', 'trapecio'],
                'difficulty' => 'avanzado',
                'is_compound' => true,
                'equipment' => 'barra',
                'instructions' => '1. Párate frente a la barra con los pies a la anchura de caderas.
2. Agarra la barra con agarre mixto o doble prono.
3. Levanta la barra manteniendo la espalda recta.
4. Extiende caderas y rodillas simultáneamente.
5. Baja la barra controladamente.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Barbell_Deadlift/0.gif',
            ],
            [
                'name' => 'Remo en Máquina',
                'description' => 'Variación en máquina para principiantes o aislamiento.',
                'category' => 'fuerza',
                'muscle_group' => 'espalda',
                'secondary_muscles' => ['biceps'],
                'difficulty' => 'principiante',
                'is_compound' => true,
                'equipment' => 'maquina',
                'instructions' => '1. Siéntate en la máquina con el pecho contra el soporte.
2. Agarra las asas con ambas manos.
3. Tira hacia atrás retrayendo los omóplatos.
4. Vuelve controladamente a la posición inicial.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Machine_Row/0.gif',
            ],

            // ================= HOMBROS =================
            [
                'name' => 'Press Militar',
                'description' => 'Ejercicio principal para el desarrollo de los hombros.',
                'category' => 'fuerza',
                'muscle_group' => 'hombros',
                'secondary_muscles' => ['triceps', 'trapecio'],
                'difficulty' => 'intermedio',
                'is_compound' => true,
                'equipment' => 'barra',
                'instructions' => '1. Párate o siéntate con la barra a la altura de los hombros.
2. Agarra la barra con agarre prono.
3. Empuja la barra hacia arriba hasta extender los brazos.
4. Baja controladamente hasta los hombros.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Barbell_Shoulder_Press/0.gif',
            ],
            [
                'name' => 'Press con Mancuernas Sentado',
                'description' => 'Versión con mancuernas para mayor rango de movimiento.',
                'category' => 'fuerza',
                'muscle_group' => 'hombros',
                'secondary_muscles' => ['triceps'],
                'difficulty' => 'intermedio',
                'is_compound' => true,
                'equipment' => 'mancuernas',
                'instructions' => '1. Siéntate con respaldo y una mancuerna en cada mano.
2. Posiciona las mancuernas a la altura de los hombros.
3. Empuja hacia arriba hasta extender los brazos.
4. Baja controladamente.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Dumbbell_Shoulder_Press/0.gif',
            ],
            [
                'name' => 'Elevaciones Laterales',
                'description' => 'Ejercicio de aislamiento para el deltoides lateral.',
                'category' => 'fuerza',
                'muscle_group' => 'hombros',
                'secondary_muscles' => [],
                'difficulty' => 'principiante',
                'is_compound' => false,
                'equipment' => 'mancuernas',
                'instructions' => '1. Párate con una mancuerna en cada mano.
2. Con ligera flexión de codos, eleva los brazos hacia los lados.
3. Sube hasta que los brazos estén paralelos al suelo.
4. Baja controladamente.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Dumbbell_Lateral_Raise/0.gif',
            ],
            [
                'name' => 'Elevaciones Frontales',
                'description' => 'Ejercicio para el deltoides anterior.',
                'category' => 'fuerza',
                'muscle_group' => 'hombros',
                'secondary_muscles' => ['pecho'],
                'difficulty' => 'principiante',
                'is_compound' => false,
                'equipment' => 'mancuernas',
                'instructions' => '1. Párate con mancuernas frente a los muslos.
2. Eleva un brazo hacia el frente hasta la altura del hombro.
3. Baja y repite con el otro brazo.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Dumbbell_Front_Raise/0.gif',
            ],
            [
                'name' => 'Pájaros (Rear Delt Fly)',
                'description' => 'Ejercicio para el deltoides posterior.',
                'category' => 'fuerza',
                'muscle_group' => 'hombros',
                'secondary_muscles' => ['trapecio'],
                'difficulty' => 'intermedio',
                'is_compound' => false,
                'equipment' => 'mancuernas',
                'instructions' => '1. Inclínate hacia adelante con mancuernas colgando.
2. Con codos ligeramente flexionados, eleva los brazos hacia los lados.
3. Aprieta los omóplatos en la parte superior.
4. Baja controladamente.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Dumbbell_Rear_Delt_Fly/0.gif',
            ],
            [
                'name' => 'Face Pull',
                'description' => 'Ejercicio correctivo y de desarrollo para deltoides posterior.',
                'category' => 'fuerza',
                'muscle_group' => 'hombros',
                'secondary_muscles' => ['trapecio'],
                'difficulty' => 'principiante',
                'is_compound' => false,
                'equipment' => 'cable',
                'instructions' => '1. Ajusta la polea a la altura de la cara.
2. Agarra la cuerda con ambas manos.
3. Tira hacia tu cara separando las manos al final.
4. Controla el regreso a la posición inicial.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Face_Pull/0.gif',
            ],

            // ================= BRAZOS (BÍCEPS) =================
            [
                'name' => 'Curl con Barra',
                'description' => 'Ejercicio básico para el desarrollo de los bíceps.',
                'category' => 'fuerza',
                'muscle_group' => 'biceps',
                'secondary_muscles' => ['antebrazos'],
                'difficulty' => 'principiante',
                'is_compound' => false,
                'equipment' => 'barra',
                'instructions' => '1. Párate con la barra en las manos, brazos extendidos.
2. Flexiona los codos para subir la barra.
3. Mantén los codos pegados al cuerpo.
4. Baja controladamente.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Barbell_Curl/0.gif',
            ],
            [
                'name' => 'Curl con Mancuernas',
                'description' => 'Curl clásico con mancuernas.',
                'category' => 'fuerza',
                'muscle_group' => 'biceps',
                'secondary_muscles' => ['antebrazos'],
                'difficulty' => 'principiante',
                'is_compound' => false,
                'equipment' => 'mancuernas',
                'instructions' => '1. Párate con una mancuerna en cada mano.
2. Flexiona los codos alternando o simultáneamente.
3. Supina las muñecas durante el movimiento.
4. Baja controladamente.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Dumbbell_Bicep_Curl/0.gif',
            ],
            [
                'name' => 'Curl Martillo',
                'description' => 'Variación que trabaja bíceps y braquial.',
                'category' => 'fuerza',
                'muscle_group' => 'biceps',
                'secondary_muscles' => ['antebrazos'],
                'difficulty' => 'principiante',
                'is_compound' => false,
                'equipment' => 'mancuernas',
                'instructions' => '1. Párate con mancuernas en agarre neutro (palmas enfrentadas).
2. Flexiona los codos manteniendo el agarre neutro.
3. Sube hasta la contracción máxima.
4. Baja controladamente.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Dumbbell_Hammer_Curl/0.gif',
            ],
            [
                'name' => 'Curl Concentrado',
                'description' => 'Ejercicio de aislamiento máximo para el bíceps.',
                'category' => 'fuerza',
                'muscle_group' => 'biceps',
                'secondary_muscles' => [],
                'difficulty' => 'principiante',
                'is_compound' => false,
                'equipment' => 'mancuernas',
                'instructions' => '1. Siéntate y apoya el codo en la cara interna del muslo.
2. Con una mancuerna, flexiona el codo.
3. Contrae el bíceps en la parte superior.
4. Baja lentamente.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Concentration_Curl/0.gif',
            ],

            // ================= BRAZOS (TRÍCEPS) =================
            [
                'name' => 'Press Francés',
                'description' => 'Ejercicio clásico para el desarrollo del tríceps.',
                'category' => 'fuerza',
                'muscle_group' => 'triceps',
                'secondary_muscles' => [],
                'difficulty' => 'intermedio',
                'is_compound' => false,
                'equipment' => 'barra',
                'instructions' => '1. Acuéstate con la barra sobre el pecho.
2. Baja la barra hacia la frente flexionando codos.
3. Extiende los codos para volver arriba.
4. Mantén los codos fijos.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Skull_Crusher/0.gif',
            ],
            [
                'name' => 'Extensión de Tríceps en Polea',
                'description' => 'Ejercicio de polea para aislar el tríceps.',
                'category' => 'fuerza',
                'muscle_group' => 'triceps',
                'secondary_muscles' => [],
                'difficulty' => 'principiante',
                'is_compound' => false,
                'equipment' => 'cable',
                'instructions' => '1. Párate frente a la polea alta.
2. Agarra la barra o cuerda con los codos al lado del cuerpo.
3. Extiende los codos empujando hacia abajo.
4. Vuelve controladamente.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Tricep_Pushdown/0.gif',
            ],
            [
                'name' => 'Fondos en Banco',
                'description' => 'Ejercicio de peso corporal para tríceps.',
                'category' => 'calistenia',
                'muscle_group' => 'triceps',
                'secondary_muscles' => ['hombros', 'pecho'],
                'difficulty' => 'principiante',
                'is_compound' => true,
                'equipment' => 'banca',
                'instructions' => '1. Siéntate en el borde del banco, manos junto a las caderas.
2. Desliza los glúteos fuera del banco.
3. Baja el cuerpo flexionando los codos a 90 grados.
4. Empuja hacia arriba extendiendo los brazos.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Bench_Dips/0.gif',
            ],
            [
                'name' => 'Extensión de Tríceps Sobre la Cabeza',
                'description' => 'Ejercicio para la cabeza larga del tríceps.',
                'category' => 'fuerza',
                'muscle_group' => 'triceps',
                'secondary_muscles' => [],
                'difficulty' => 'intermedio',
                'is_compound' => false,
                'equipment' => 'mancuernas',
                'instructions' => '1. Sostén una mancuerna sobre la cabeza con ambas manos.
2. Baja la mancuerna detrás de la cabeza flexionando codos.
3. Extiende los codos para subir.
4. Mantén los codos cerca de las orejas.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Dumbbell_Overhead_Tricep_Extension/0.gif',
            ],

            // ================= PIERNAS (CUÁDRICEPS) =================
            [
                'name' => 'Sentadilla con Barra',
                'description' => 'Ejercicio rey para el desarrollo de las piernas.',
                'category' => 'fuerza',
                'muscle_group' => 'cuadriceps',
                'secondary_muscles' => ['gluteos', 'isquiotibiales'],
                'difficulty' => 'intermedio',
                'is_compound' => true,
                'equipment' => 'barra',
                'instructions' => '1. Coloca la barra sobre los trapecios.
2. Separa los pies al ancho de hombros.
3. Baja flexionando caderas y rodillas.
4. Mantén la espalda recta y el pecho arriba.
5. Sube empujando desde los talones.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Barbell_Squat/0.gif',
            ],
            [
                'name' => 'Prensa de Piernas',
                'description' => 'Ejercicio en máquina para desarrollar cuádriceps.',
                'category' => 'fuerza',
                'muscle_group' => 'cuadriceps',
                'secondary_muscles' => ['gluteos'],
                'difficulty' => 'principiante',
                'is_compound' => true,
                'equipment' => 'maquina',
                'instructions' => '1. Siéntate en la máquina con la espalda apoyada.
2. Coloca los pies en la plataforma al ancho de hombros.
3. Baja el peso flexionando las rodillas.
4. Empuja hasta casi extender las piernas.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Leg_Press/0.gif',
            ],
            [
                'name' => 'Extensión de Cuádriceps',
                'description' => 'Ejercicio de aislamiento para el cuádriceps.',
                'category' => 'fuerza',
                'muscle_group' => 'cuadriceps',
                'secondary_muscles' => [],
                'difficulty' => 'principiante',
                'is_compound' => false,
                'equipment' => 'maquina',
                'instructions' => '1. Siéntate en la máquina con las piernas detrás del rodillo.
2. Extiende las piernas hasta que estén rectas.
3. Contrae el cuádriceps en la parte superior.
4. Baja controladamente.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Leg_Extension/0.gif',
            ],
            [
                'name' => 'Zancadas',
                'description' => 'Ejercicio funcional unilateral para piernas.',
                'category' => 'fuerza',
                'muscle_group' => 'cuadriceps',
                'secondary_muscles' => ['gluteos', 'isquiotibiales'],
                'difficulty' => 'intermedio',
                'is_compound' => true,
                'equipment' => 'mancuernas',
                'instructions' => '1. Párate con mancuernas en las manos.
2. Da un paso adelante y baja la rodilla trasera.
3. La rodilla delantera no debe pasar el pie.
4. Empuja desde el talón para volver.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Dumbbell_Lunge/0.gif',
            ],
            [
                'name' => 'Sentadilla Búlgara',
                'description' => 'Ejercicio unilateral avanzado para piernas.',
                'category' => 'fuerza',
                'muscle_group' => 'cuadriceps',
                'secondary_muscles' => ['gluteos'],
                'difficulty' => 'avanzado',
                'is_compound' => true,
                'equipment' => 'mancuernas',
                'instructions' => '1. Coloca un pie atrás sobre un banco.
2. Baja flexionando la rodilla delantera.
3. Mantén el torso erguido.
4. Empuja para volver arriba.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Bulgarian_Split_Squat/0.gif',
            ],
            [
                'name' => 'Sentadilla Goblet',
                'description' => 'Variación de sentadilla ideal para principiantes.',
                'category' => 'fuerza',
                'muscle_group' => 'cuadriceps',
                'secondary_muscles' => ['gluteos', 'abdominales'],
                'difficulty' => 'principiante',
                'is_compound' => true,
                'equipment' => 'kettlebell',
                'instructions' => '1. Sostén una kettlebell o mancuerna contra el pecho.
2. Separa los pies ligeramente más que el ancho de hombros.
3. Baja en sentadilla manteniendo el pecho arriba.
4. Sube empujando desde los talones.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Goblet_Squat/0.gif',
            ],

            // ================= PIERNAS (ISQUIOTIBIALES/GLÚTEOS) =================
            [
                'name' => 'Peso Muerto Rumano',
                'description' => 'Ejercicio para isquiotibiales y glúteos.',
                'category' => 'fuerza',
                'muscle_group' => 'isquiotibiales',
                'secondary_muscles' => ['gluteos', 'espalda'],
                'difficulty' => 'intermedio',
                'is_compound' => true,
                'equipment' => 'barra',
                'instructions' => '1. Párate con la barra frente a los muslos.
2. Con las rodillas ligeramente flexionadas, baja la barra.
3. Empuja las caderas hacia atrás.
4. Siente el estiramiento en isquiotibiales y sube.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Romanian_Deadlift/0.gif',
            ],
            [
                'name' => 'Curl de Piernas Acostado',
                'description' => 'Ejercicio de aislamiento para isquiotibiales.',
                'category' => 'fuerza',
                'muscle_group' => 'isquiotibiales',
                'secondary_muscles' => [],
                'difficulty' => 'principiante',
                'is_compound' => false,
                'equipment' => 'maquina',
                'instructions' => '1. Acuéstate boca abajo en la máquina.
2. Coloca los tobillos bajo el rodillo.
3. Flexiona las rodillas llevando los talones a los glúteos.
4. Baja controladamente.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Lying_Leg_Curl/0.gif',
            ],
            [
                'name' => 'Hip Thrust',
                'description' => 'Ejercicio principal para el desarrollo de glúteos.',
                'category' => 'fuerza',
                'muscle_group' => 'gluteos',
                'secondary_muscles' => ['isquiotibiales'],
                'difficulty' => 'intermedio',
                'is_compound' => false,
                'equipment' => 'barra',
                'instructions' => '1. Siéntate en el suelo con la espalda contra un banco.
2. Coloca la barra sobre las caderas.
3. Empuja las caderas hacia arriba contrayendo glúteos.
4. Baja controladamente.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Hip_Thrust/0.gif',
            ],
            [
                'name' => 'Puente de Glúteos',
                'description' => 'Ejercicio básico para activación de glúteos.',
                'category' => 'fuerza',
                'muscle_group' => 'gluteos',
                'secondary_muscles' => ['isquiotibiales'],
                'difficulty' => 'principiante',
                'is_compound' => false,
                'equipment' => 'ninguno',
                'instructions' => '1. Acuéstate boca arriba con rodillas flexionadas.
2. Pies apoyados en el suelo al ancho de caderas.
3. Eleva las caderas apretando los glúteos.
4. Baja controladamente.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Glute_Bridge/0.gif',
            ],

            // ================= PIERNAS (PANTORRILLAS) =================
            [
                'name' => 'Elevación de Talones de Pie',
                'description' => 'Ejercicio principal para las pantorrillas.',
                'category' => 'fuerza',
                'muscle_group' => 'pantorrillas',
                'secondary_muscles' => [],
                'difficulty' => 'principiante',
                'is_compound' => false,
                'equipment' => 'maquina',
                'instructions' => '1. Párate en la máquina con los hombros bajo las almohadillas.
2. Coloca los pies en el borde de la plataforma.
3. Elévate sobre las puntas de los pies.
4. Baja estirando las pantorrillas.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Standing_Calf_Raise/0.gif',
            ],
            [
                'name' => 'Elevación de Talones Sentado',
                'description' => 'Ejercicio para el sóleo.',
                'category' => 'fuerza',
                'muscle_group' => 'pantorrillas',
                'secondary_muscles' => [],
                'difficulty' => 'principiante',
                'is_compound' => false,
                'equipment' => 'maquina',
                'instructions' => '1. Siéntate en la máquina con las rodillas bajo la almohadilla.
2. Coloca los pies en el borde de la plataforma.
3. Eleva los talones contrayendo las pantorrillas.
4. Baja estirando.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Seated_Calf_Raise/0.gif',
            ],

            // ================= ABDOMINALES =================
            [
                'name' => 'Crunch Abdominal',
                'description' => 'Ejercicio básico para los abdominales superiores.',
                'category' => 'fuerza',
                'muscle_group' => 'abdominales',
                'secondary_muscles' => [],
                'difficulty' => 'principiante',
                'is_compound' => false,
                'equipment' => 'ninguno',
                'instructions' => '1. Acuéstate boca arriba con rodillas flexionadas.
2. Manos detrás de la cabeza o cruzadas sobre el pecho.
3. Eleva los hombros del suelo contrayendo abdominales.
4. Baja controladamente.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Crunch/0.gif',
            ],
            [
                'name' => 'Plancha',
                'description' => 'Ejercicio isométrico para el core completo.',
                'category' => 'calistenia',
                'muscle_group' => 'abdominales',
                'secondary_muscles' => ['oblicuos', 'espalda'],
                'difficulty' => 'principiante',
                'is_compound' => true,
                'equipment' => 'ninguno',
                'instructions' => '1. Apóyate sobre antebrazos y puntas de los pies.
2. Mantén el cuerpo en línea recta.
3. Contrae el abdomen y glúteos.
4. Mantén la posición el tiempo indicado.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Plank/0.gif',
            ],
            [
                'name' => 'Elevación de Piernas Colgado',
                'description' => 'Ejercicio avanzado para abdominales inferiores.',
                'category' => 'calistenia',
                'muscle_group' => 'abdominales',
                'secondary_muscles' => ['oblicuos'],
                'difficulty' => 'avanzado',
                'is_compound' => false,
                'equipment' => 'barra_dominadas',
                'instructions' => '1. Cuélgate de una barra con agarre prono.
2. Eleva las piernas hasta formar 90 grados.
3. Mantén las piernas rectas.
4. Baja controladamente.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Hanging_Leg_Raise/0.gif',
            ],
            [
                'name' => 'Russian Twist',
                'description' => 'Ejercicio rotacional para oblicuos.',
                'category' => 'fuerza',
                'muscle_group' => 'oblicuos',
                'secondary_muscles' => ['abdominales'],
                'difficulty' => 'intermedio',
                'is_compound' => false,
                'equipment' => 'ninguno',
                'instructions' => '1. Siéntate con las rodillas flexionadas, pies elevados.
2. Inclina el torso ligeramente hacia atrás.
3. Rota el torso llevando las manos de lado a lado.
4. Mantén el core contraído.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Russian_Twist/0.gif',
            ],
            [
                'name' => 'Mountain Climbers',
                'description' => 'Ejercicio dinámico para core y cardio.',
                'category' => 'hiit',
                'muscle_group' => 'abdominales',
                'secondary_muscles' => ['hombros', 'cuadriceps'],
                'difficulty' => 'intermedio',
                'is_compound' => true,
                'equipment' => 'ninguno',
                'instructions' => '1. Posición de plancha alta con brazos extendidos.
2. Lleva una rodilla al pecho.
3. Alterna rápidamente las piernas.
4. Mantén las caderas bajas.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Mountain_Climber/0.gif',
            ],
            [
                'name' => 'Dead Bug',
                'description' => 'Ejercicio correctivo para estabilización del core.',
                'category' => 'funcional',
                'muscle_group' => 'abdominales',
                'secondary_muscles' => [],
                'difficulty' => 'principiante',
                'is_compound' => false,
                'equipment' => 'ninguno',
                'instructions' => '1. Acuéstate boca arriba con brazos hacia el techo.
2. Rodillas flexionadas a 90 grados.
3. Extiende brazo y pierna opuestos.
4. Mantén la espalda baja pegada al suelo.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Dead_Bug/0.gif',
            ],

            // ================= CARDIO =================
            [
                'name' => 'Burpees',
                'description' => 'Ejercicio de cuerpo completo y alta intensidad.',
                'category' => 'hiit',
                'muscle_group' => 'cuerpo_completo',
                'secondary_muscles' => ['pecho', 'cuadriceps', 'abdominales'],
                'difficulty' => 'avanzado',
                'is_compound' => true,
                'equipment' => 'ninguno',
                'instructions' => '1. Desde de pie, baja a posición de sentadilla.
2. Coloca las manos en el suelo y salta a plancha.
3. Haz una flexión (opcional).
4. Salta los pies hacia las manos y salta arriba.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Burpee/0.gif',
            ],
            [
                'name' => 'Jumping Jacks',
                'description' => 'Ejercicio cardiovascular básico.',
                'category' => 'cardio',
                'muscle_group' => 'cuerpo_completo',
                'secondary_muscles' => [],
                'difficulty' => 'principiante',
                'is_compound' => true,
                'equipment' => 'ninguno',
                'instructions' => '1. Párate con pies juntos y brazos a los lados.
2. Salta separando los pies y elevando los brazos.
3. Salta para volver a la posición inicial.
4. Repite de forma continua.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Jumping_Jack/0.gif',
            ],
            [
                'name' => 'Salto de Cuerda',
                'description' => 'Cardio clásico para coordinación y resistencia.',
                'category' => 'cardio',
                'muscle_group' => 'cuerpo_completo',
                'secondary_muscles' => ['pantorrillas'],
                'difficulty' => 'intermedio',
                'is_compound' => true,
                'equipment' => 'cuerda',
                'instructions' => '1. Sostén la cuerda con ambas manos.
2. Gira la cuerda con las muñecas.
3. Salta sobre la cuerda con ambos pies.
4. Mantén un ritmo constante.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Jump_Rope/0.gif',
            ],
            [
                'name' => 'High Knees',
                'description' => 'Ejercicio de cardio que trabaja el core.',
                'category' => 'cardio',
                'muscle_group' => 'cuerpo_completo',
                'secondary_muscles' => ['cuadriceps', 'abdominales'],
                'difficulty' => 'principiante',
                'is_compound' => true,
                'equipment' => 'ninguno',
                'instructions' => '1. Corre en el lugar levantando las rodillas alto.
2. Las rodillas deben llegar a la altura de la cadera.
3. Balancea los brazos de forma coordinada.
4. Mantén un ritmo rápido.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/High_Knees/0.gif',
            ],
            [
                'name' => 'Box Jumps',
                'description' => 'Ejercicio pliométrico para potencia de piernas.',
                'category' => 'pliometria',
                'muscle_group' => 'cuadriceps',
                'secondary_muscles' => ['gluteos', 'pantorrillas'],
                'difficulty' => 'intermedio',
                'is_compound' => true,
                'equipment' => 'banca',
                'instructions' => '1. Párate frente a una caja o banco estable.
2. Flexiona las rodillas y balancea los brazos.
3. Salta explosivamente sobre la caja.
4. Aterriza suavemente y baja con control.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Box_Jump/0.gif',
            ],

            // ================= ESTIRAMIENTOS =================
            [
                'name' => 'Estiramiento de Cuádriceps',
                'description' => 'Estiramiento básico para los cuádriceps.',
                'category' => 'flexibilidad',
                'muscle_group' => 'cuadriceps',
                'secondary_muscles' => [],
                'difficulty' => 'principiante',
                'is_compound' => false,
                'equipment' => 'ninguno',
                'instructions' => '1. Párate sobre una pierna.
2. Flexiona la otra rodilla y agarra el tobillo.
3. Lleva el talón hacia el glúteo.
4. Mantén la posición 20-30 segundos.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Standing_Quad_Stretch/0.gif',
            ],
            [
                'name' => 'Estiramiento de Isquiotibiales',
                'description' => 'Estiramiento para la parte posterior del muslo.',
                'category' => 'flexibilidad',
                'muscle_group' => 'isquiotibiales',
                'secondary_muscles' => [],
                'difficulty' => 'principiante',
                'is_compound' => false,
                'equipment' => 'ninguno',
                'instructions' => '1. Siéntate con una pierna extendida.
2. Flexiona la otra pierna con el pie en el muslo interno.
3. Inclínate hacia adelante desde la cadera.
4. Mantén 20-30 segundos.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Seated_Hamstring_Stretch/0.gif',
            ],
            [
                'name' => 'Estiramiento de Pecho en Puerta',
                'description' => 'Estiramiento para abrir el pecho.',
                'category' => 'flexibilidad',
                'muscle_group' => 'pecho',
                'secondary_muscles' => ['hombros'],
                'difficulty' => 'principiante',
                'is_compound' => false,
                'equipment' => 'ninguno',
                'instructions' => '1. Párate en el marco de una puerta.
2. Coloca los antebrazos en los lados del marco.
3. Da un paso adelante estirando el pecho.
4. Mantén 20-30 segundos.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Doorway_Chest_Stretch/0.gif',
            ],
            [
                'name' => 'Cat-Cow Stretch',
                'description' => 'Estiramiento dinámico para la columna.',
                'category' => 'flexibilidad',
                'muscle_group' => 'espalda',
                'secondary_muscles' => ['abdominales'],
                'difficulty' => 'principiante',
                'is_compound' => false,
                'equipment' => 'ninguno',
                'instructions' => '1. Posición de cuatro puntos (manos y rodillas).
2. Arquea la espalda hacia arriba (gato).
3. Luego arquea hacia abajo mirando arriba (vaca).
4. Alterna fluidamente.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Cat_Cow_Stretch/0.gif',
            ],
            [
                'name' => 'Hip Flexor Stretch',
                'description' => 'Estiramiento para los flexores de cadera.',
                'category' => 'flexibilidad',
                'muscle_group' => 'cuadriceps',
                'secondary_muscles' => ['gluteos'],
                'difficulty' => 'principiante',
                'is_compound' => false,
                'equipment' => 'ninguno',
                'instructions' => '1. Arrodíllate con una pierna adelante en ángulo recto.
2. Mantén el torso erguido.
3. Empuja las caderas hacia adelante.
4. Mantén 20-30 segundos por lado.',
                'gif_url' => 'https://raw.githubusercontent.com/yuhonas/free-exercise-db/main/exercises/Hip_Flexor_Stretch/0.gif',
            ],
        ];

        foreach ($exercises as $exercise) {
            ExerciseLibrary::updateOrCreate(
                ['name' => $exercise['name'], 'coach_id' => null],
                array_merge($exercise, [
                    'coach_id' => null,
                    'is_active' => true,
                    'is_system' => true,
                ])
            );
        }
    }
}
