<?php

namespace Database\Seeders;

use App\Models\Client;
use App\Models\User;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\Hash;

/**
 * TEST/DEVELOPMENT ONLY - DO NOT RUN IN PRODUCTION
 * 
 * This seeder creates test users with known passwords for local development.
 * It should NEVER be run in production environments.
 * 
 * Usage: php artisan db:seed --class=ClientPortalTestSeeder
 */
class ClientPortalTestSeeder extends Seeder
{
    /**
     * Create a test client with portal access for testing the Client Portal.
     * 
     * WARNING: This seeder creates users with weak, known passwords.
     * Only use in development/testing environments.
     */
    public function run(): void
    {
        // Safety check - prevent running in production
        if (app()->environment('production')) {
            $this->command->error('⚠️  This seeder cannot be run in production!');
            return;
        }

        // First, find or create a coach user
        $coach = User::where('role', 'coach')->where('status', 'active')->first();
        
        if (!$coach) {
            $coach = User::create([
                'name' => 'Coach Demo',
                'email' => 'coach@demo.com',
                'password' => Hash::make('password123'),
                'role' => 'coach',
                'status' => 'active',
                'email_verified_at' => now(),
            ]);
            
            $this->command->info('Created demo coach: coach@demo.com / password123');
        }

        // Check if test client portal user already exists
        $existingClientUser = User::where('email', 'cliente@test.com')->first();
        
        if ($existingClientUser) {
            $this->command->warn('Test client user already exists: cliente@test.com');
            return;
        }

        // Create the client portal user account
        $clientUser = User::create([
            'name' => 'Cliente de Prueba',
            'email' => 'cliente@test.com',
            'password' => Hash::make('cliente123'),
            'role' => 'client',
            'status' => 'active',
            'email_verified_at' => now(),
        ]);

        // Create the client record linked to the coach and portal user
        $client = Client::create([
            'user_id' => $coach->id, // belongs to coach
            'client_user_id' => $clientUser->id, // portal access
            'name' => 'Cliente de Prueba',
            'email' => 'cliente@test.com',
            'phone' => '+1 809 555 1234',
            'age' => 28,
            'weight' => 75, // kg
            'height' => 175, // cm
            'gender' => 'male',
            'objective' => 'maintain',
            'activity_level' => 'moderate',
            'notes' => 'Usuario de prueba para testear el Portal del Cliente.',
            'is_active' => true,
            'portal_active' => true,
        ]);

        // Calculate and save metrics
        $client->updateMetrics();

        $this->command->info('');
        $this->command->info('===========================================');
        $this->command->info('   USUARIO DE PRUEBA PORTAL DEL CLIENTE   ');
        $this->command->info('===========================================');
        $this->command->info('');
        $this->command->info('📧 Email: cliente@test.com');
        $this->command->info('🔑 Password: cliente123');
        $this->command->info('');
        $this->command->info('Datos del Cliente:');
        $this->command->info('- Nombre: ' . $client->name);
        $this->command->info('- Edad: ' . $client->age . ' años');
        $this->command->info('- Peso: ' . $client->weight . ' kg');
        $this->command->info('- Altura: ' . $client->height . ' cm');
        $this->command->info('- IMC: ' . $client->bmi);
        $this->command->info('- TMB: ' . $client->bmr . ' kcal');
        $this->command->info('- TDEE: ' . $client->tdee . ' kcal');
        $this->command->info('');
        $this->command->info('El cliente pertenece al coach: ' . $coach->email);
        $this->command->info('===========================================');
    }
}
