<?php

namespace App\Services;

use App\Models\Client;
use App\Models\Invoice;
use App\Models\MealPlan;
use App\Models\PdfSetting;
use App\Models\Setting;
use App\Models\User;
use App\Models\Workout;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Spatie\Browsershot\Browsershot;

class PdfBrowsershotService
{
    /**
     * Get PDF settings for a specific document type.
     */
    private function getPdfSettings(string $documentType): array
    {
        $settings = PdfSetting::getSettingsForType($documentType);
        
        return [
            'pdf_settings' => $settings,
            'primary_color' => $settings['primary_color'] ?? '#3B82F6',
            'secondary_color' => $settings['secondary_color'] ?? '#F97316',
            'accent_color' => $settings['accent_color'] ?? '#10B981',
            'show_logo' => $settings['show_logo'] ?? true,
            'show_date' => $settings['show_date'] ?? true,
            'show_page_numbers' => $settings['show_page_numbers'] ?? true,
            'show_coach_info' => $settings['show_coach_info'] ?? true,
            'header_title' => $settings['header_title'] ?? null,
            'footer_text' => $settings['footer_text'] ?? null,
            'logo' => $settings['logo'] ?? Setting::get('logo'),
            'site_name' => Setting::get('site_name', 'CoachPro'),
        ];
    }

    /**
     * Get coach information for PDFs.
     */
    private function getCoachInfo(?User $coach): array
    {
        if (!$coach) {
            return [];
        }

        // Try multiple photo sources
        $photoPath = null;
        if ($coach->profile_photo_path) {
            $photoPath = asset('storage/' . $coach->profile_photo_path);
        } elseif ($coach->avatar ?? null) {
            $avatar = $coach->avatar;
            $photoPath = str_starts_with($avatar, 'http') 
                ? $avatar 
                : asset('storage/' . $avatar);
        }

        return [
            'name' => $coach->name,
            'email' => $coach->email,
            'phone' => $coach->phone ?? null,
            'photo' => $photoPath,
            'credentials' => $coach->credentials ?? $coach->bio ?? null,
            'website' => $coach->website ?? null,
            'specialty' => $coach->specialty ?? null,
        ];
    }


    /**
     * Create Browsershot instance with common settings.
     */
    private function createBrowsershot(string $html): Browsershot
    {
        $browsershot = Browsershot::html($html)
            ->format('A4')
            ->margins(10, 10, 15, 10)
            ->showBackground()
            ->waitUntilNetworkIdle()
            ->timeout(60);

        // Set Node and NPM paths for Windows
        if (PHP_OS_FAMILY === 'Windows') {
            $npmPath = base_path('node_modules/.bin');
            $browsershot->setNodeBinary('node')
                ->setNpmBinary('npm');
        }

        return $browsershot;
    }

    /**
     * Generate modern meal plan PDF.
     */
    public function generateMealPlan(MealPlan $mealPlan, string $lang = 'es'): string
    {
        $mealPlan->load(['client.user', 'client.portalUser', 'dietPlans.mealItems.food']);
        
        $coach = $mealPlan->client?->user;

        if (!$mealPlan->client) {
            throw new \Exception('Meal plan has no associated client');
        }

        $client = $mealPlan->client;

        // Build client avatar as base64 data URI for PDF rendering
        // Browsershot headless Chrome cannot access localhost URLs, so we embed the image
        $clientAvatarUrl = null;
        
        // Get the avatar path from portalUser (User model) or client
        $avatarPath = null;
        if ($client->portalUser && $client->portalUser->avatar) {
            $avatarPath = $client->portalUser->avatar;
        } elseif ($client->avatar) {
            $avatarPath = $client->avatar;
        }
        
        if ($avatarPath) {
            // If it's already a full URL (e.g. Google avatar), use directly
            if (str_starts_with($avatarPath, 'http')) {
                // Try to download and convert to base64
                try {
                    $imageContent = @file_get_contents($avatarPath);
                    if ($imageContent) {
                        $mimeType = 'image/jpeg';
                        $finfo = new \finfo(FILEINFO_MIME_TYPE);
                        $detectedMime = $finfo->buffer($imageContent);
                        if ($detectedMime) {
                            $mimeType = $detectedMime;
                        }
                        $clientAvatarUrl = 'data:' . $mimeType . ';base64,' . base64_encode($imageContent);
                    }
                } catch (\Exception $e) {
                    // Fallback: use the URL directly
                    $clientAvatarUrl = $avatarPath;
                }
            } else {
                // It's a storage path - read from local filesystem
                $storagePath = storage_path('app/public/' . $avatarPath);
                if (file_exists($storagePath)) {
                    $imageContent = file_get_contents($storagePath);
                    $mimeType = mime_content_type($storagePath) ?: 'image/jpeg';
                    $clientAvatarUrl = 'data:' . $mimeType . ';base64,' . base64_encode($imageContent);
                } else {
                    // Try with asset() as last resort
                    $clientAvatarUrl = asset('storage/' . $avatarPath);
                }
            }
        }
        
        // Set on client object so template can access it
        $client->avatar_url = $clientAvatarUrl;

        // Calculate totals from pre-calculated MealItem values
        $totals = ['proteins' => 0, 'carbs' => 0, 'fats' => 0, 'calories' => 0];
        
        foreach ($mealPlan->dietPlans as $dietPlan) {
            foreach ($dietPlan->mealItems as $item) {
                $totals['calories'] += $item->calories ?? 0;
                $totals['proteins'] += $item->proteins ?? 0;
                $totals['carbs'] += $item->carbs ?? 0;
                $totals['fats'] += $item->fats ?? 0;
            }
        }

        try {
            $settings = $this->getPdfSettings(PdfSetting::TYPE_MEAL_PLAN);
            $coachInfo = $this->getCoachInfo($coach);

            // Use the new meal-plan-new template
            $html = view('pdf.meal-plan-new', [
                'mealPlan' => $mealPlan,
                'client' => $client,
                'dietPlans' => $mealPlan->dietPlans,
                'totals' => $totals,
                'mealLabels' => $this->getMealLabels(),
                'settings' => $settings,
                'coach' => $coachInfo,
                'lang' => $lang,
                'translations' => [],
            ])->render();

            $directory = 'meal-plans';
            $filename = "{$directory}/dieta-{$mealPlan->client->id}-{$mealPlan->id}.pdf";

            $disk = Storage::disk('public');
            if (!$disk->exists($directory)) {
                $disk->makeDirectory($directory);
            }

            $fullPath = $disk->path($filename);
            
            $this->createBrowsershot($html)->save($fullPath);

            Log::info("Modern PDF generated with new template: {$filename}");

            return $filename;

        } catch (\Exception $e) {
            Log::error('Browsershot PDF Error: ' . $e->getMessage());
            throw $e;
        }
    }


    /**
     * Generate modern workout PDF.
     */
    public function generateWorkout(Workout $workout): string
    {
        $workout->load(['client.user', 'exercises']);
        
        $coach = $workout->client?->user;

        if (!$workout->client) {
            throw new \Exception('Workout must have an associated client');
        }

        try {
            $settings = $this->getPdfSettings(PdfSetting::TYPE_WORKOUT);
            $coachInfo = $this->getCoachInfo($coach);

            $html = view('pdf.modern.workout', [
                'workout' => $workout,
                'client' => $workout->client,
                'exercises' => $workout->exercises->sortBy('order'),
                'days' => $this->getDayLabels($workout->days ?? []),
                'settings' => $settings,
                'coach' => $coachInfo,
            ])->render();

            $directory = 'workouts';
            $filename = "{$directory}/rutina-{$workout->client->id}-{$workout->id}.pdf";

            $disk = Storage::disk('public');
            if (!$disk->exists($directory)) {
                $disk->makeDirectory($directory);
            }

            $fullPath = $disk->path($filename);

            $this->createBrowsershot($html)->save($fullPath);

            Log::info("Modern workout PDF generated: {$filename}");

            return $filename;

        } catch (\Exception $e) {
            Log::error('Browsershot Workout PDF Error: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Generate modern invoice PDF.
     */
    public function generateInvoice(Invoice $invoice): string
    {
        $invoice->load('user', 'subscription.plan');
        
        try {
            $settings = $this->getPdfSettings(PdfSetting::TYPE_INVOICE);
            
            // Handle items - could be JSON string or already decoded array
            $items = $invoice->items;
            if (is_string($items)) {
                $items = json_decode($items, true) ?? [];
            }

            $html = view('pdf.modern.invoice', [
                'invoice' => $invoice,
                'items' => $items,
                'settings' => $settings,
            ])->render();

            $directory = 'invoices';
            $filename = "{$directory}/{$invoice->invoice_number}.pdf";

            $disk = Storage::disk('public');
            if (!$disk->exists($directory)) {
                $disk->makeDirectory($directory);
            }

            $fullPath = $disk->path($filename);

            $this->createBrowsershot($html)->save($fullPath);

            $invoice->update(['pdf_path' => $filename]);

            Log::info("Modern invoice PDF generated: {$filename}");

            return $filename;

        } catch (\Exception $e) {
            Log::error('Browsershot Invoice PDF Error: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Get day labels in Spanish.
     */
    private function getDayLabels(array $days): array
    {
        $labels = [
            'monday' => 'Lunes',
            'tuesday' => 'Martes',
            'wednesday' => 'Miércoles',
            'thursday' => 'Jueves',
            'friday' => 'Viernes',
            'saturday' => 'Sábado',
            'sunday' => 'Domingo',
        ];

        return array_map(fn($day) => $labels[$day] ?? $day, $days);
    }

    /**
     * Get meal type labels in Spanish.
     */
    private function getMealLabels(): array
    {
        return [
            'breakfast' => 'Desayuno',
            'mid_morning_snack' => 'Media Mañana',
            'lunch' => 'Almuerzo',
            'mid_afternoon_snack' => 'Merienda',
            'dinner' => 'Cena',
            'snack' => 'Snack',
            'pre_workout' => 'Pre-entreno',
            'post_workout' => 'Post-entreno',
        ];
    }
}
