<?php

namespace App\Services;

use App\Models\Invoice;
use App\Models\Setting;
use Illuminate\Support\Facades\Log;
use Stripe\Stripe;
use Stripe\Checkout\Session as StripeSession;
use Stripe\Exception\ApiErrorException;
use Srmklive\PayPal\Services\PayPal as PayPalClient;

class PaymentService
{
    protected ?PayPalClient $paypal = null;
    protected bool $stripeInitialized = false;
    
    /**
     * Initialize Stripe with API key from settings.
     */
    protected function initStripe(): bool
    {
        if ($this->stripeInitialized) {
            return true;
        }
        
        $stripeEnabled = Setting::get('stripe_enabled', false);
        if (!$stripeEnabled) {
            Log::warning('Stripe: Payment method is disabled in settings');
            return false;
        }
        
        $stripeSecret = Setting::get('stripe_secret');
        
        if (!$stripeSecret) {
            Log::error('Stripe: No secret key configured in settings');
            return false;
        }
        
        try {
            $decryptedKey = null;
            
            // Try to decrypt the key (it's stored encrypted)
            try {
                $decryptedKey = decrypt($stripeSecret);
                Log::info('Stripe: Secret key decrypted successfully');
            } catch (\Exception $e) {
                // If decryption fails, try using the value directly (for backwards compatibility)
                Log::warning('Stripe: Could not decrypt secret, trying direct value', ['error' => $e->getMessage()]);
                $decryptedKey = $stripeSecret;
            }
            
            if (empty($decryptedKey)) {
                Log::error('Stripe: Empty secret key after processing');
                return false;
            }
            
            // Validate the key format (sk_test_ or sk_live_)
            if (!str_starts_with($decryptedKey, 'sk_')) {
                Log::error('Stripe: Invalid secret key format', ['prefix' => substr($decryptedKey, 0, 10)]);
                return false;
            }
            
            Stripe::setApiKey($decryptedKey);
            $this->stripeInitialized = true;
            
            Log::info('Stripe: Successfully initialized', [
                'mode' => str_starts_with($decryptedKey, 'sk_test_') ? 'test' : 'live'
            ]);
            return true;
        } catch (\Exception $e) {
            Log::error('Stripe initialization error: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Initialize PayPal client.
     */
    protected function initPayPal(): PayPalClient
    {
        if ($this->paypal) {
            return $this->paypal;
        }
        
        $paypalEnabled = Setting::get('paypal_enabled', false);
        if (!$paypalEnabled) {
            throw new \Exception('PayPal: El método de pago está deshabilitado en la configuración.');
        }
        
        $paypalClientId = Setting::get('paypal_client_id');
        $paypalSecret = Setting::get('paypal_secret');
        $sandbox = Setting::get('paypal_sandbox', true);
        
        // Validate credentials
        if (empty($paypalClientId)) {
            throw new \Exception('PayPal: El Client ID no está configurado. Configure las credenciales en Configuración > Pagos.');
        }
        
        if (empty($paypalSecret)) {
            throw new \Exception('PayPal: El Secret no está configurado. Configure las credenciales en Configuración > Pagos.');
        }
        
        // Try to decrypt the secret (it's stored encrypted)
        $decryptedSecret = $paypalSecret;
        try {
            $decryptedSecret = decrypt($paypalSecret);
            Log::info('PayPal: Secret decrypted successfully');
        } catch (\Exception $e) {
            // Use as-is if decryption fails (backwards compatibility)
            Log::warning('PayPal: Could not decrypt secret, using direct value', ['error' => $e->getMessage()]);
        }
        
        // Log credential validation (partial for security)
        Log::info('PayPal: Initializing', [
            'sandbox' => $sandbox,
            'client_id_length' => strlen($paypalClientId),
            'client_id_prefix' => substr($paypalClientId, 0, 15) . '...',
            'has_secret' => !empty($decryptedSecret),
            'secret_length' => strlen($decryptedSecret),
        ]);
        
        $config = [
            'mode' => $sandbox ? 'sandbox' : 'live',
            'sandbox' => [
                'client_id' => $paypalClientId,
                'client_secret' => $decryptedSecret,
                'app_id' => '',
            ],
            'live' => [
                'client_id' => $paypalClientId,
                'client_secret' => $decryptedSecret,
                'app_id' => '',
            ],
            'payment_action' => 'Sale',
            'currency' => Setting::get('currency', 'USD'),
            'notify_url' => '',
            'locale' => 'es_ES',
            'validate_ssl' => !$sandbox,
        ];
        
        try {
            $this->paypal = new PayPalClient($config);
            $this->paypal->setApiCredentials($config);
            
            $accessToken = $this->paypal->getAccessToken();
            
            if (empty($accessToken)) {
                throw new \Exception('No se pudo obtener el token de acceso de PayPal. Verifique las credenciales.');
            }
            
            Log::info('PayPal: Successfully initialized', [
                'has_access_token' => !empty($accessToken),
            ]);
            
            return $this->paypal;
        } catch (\Exception $e) {
            Log::error('PayPal initialization error', [
                'message' => $e->getMessage(),
                'sandbox' => $sandbox,
            ]);
            throw new \Exception('Error al conectar con PayPal: ' . $e->getMessage());
        }
    }
    
    /**
     * Check if Stripe is enabled.
     */
    public function isStripeEnabled(): bool
    {
        return (bool) Setting::get('stripe_enabled', false);
    }
    
    /**
     * Check if PayPal is enabled.
     */
    public function isPayPalEnabled(): bool
    {
        return (bool) Setting::get('paypal_enabled', false);
    }
    
    /**
     * Check if bank transfer is enabled.
     */
    public function isBankTransferEnabled(): bool
    {
        return (bool) Setting::get('bank_transfer_enabled', true);
    }
    
    /**
     * Verify payment gateway configuration.
     * Returns array with status and any issues found.
     */
    public function verifyConfiguration(): array
    {
        $issues = [];
        $status = [
            'stripe' => ['enabled' => false, 'configured' => false, 'error' => null],
            'paypal' => ['enabled' => false, 'configured' => false, 'error' => null],
            'bank_transfer' => ['enabled' => false, 'configured' => true, 'error' => null],
        ];
        
        // Check Stripe
        $status['stripe']['enabled'] = $this->isStripeEnabled();
        if ($status['stripe']['enabled']) {
            $stripeKey = Setting::get('stripe_key');
            $stripeSecret = Setting::get('stripe_secret');
            
            if (empty($stripeKey)) {
                $status['stripe']['error'] = 'Falta la clave pública de Stripe';
            } elseif (empty($stripeSecret)) {
                $status['stripe']['error'] = 'Falta la clave secreta de Stripe';
            } else {
                $status['stripe']['configured'] = true;
            }
        }
        
        // Check PayPal
        $status['paypal']['enabled'] = $this->isPayPalEnabled();
        if ($status['paypal']['enabled']) {
            $paypalClientId = Setting::get('paypal_client_id');
            $paypalSecret = Setting::get('paypal_secret');
            
            if (empty($paypalClientId)) {
                $status['paypal']['error'] = 'Falta el Client ID de PayPal';
            } elseif (empty($paypalSecret)) {
                $status['paypal']['error'] = 'Falta el Secret de PayPal';
            } else {
                $status['paypal']['configured'] = true;
            }
        }
        
        // Check Bank Transfer
        $status['bank_transfer']['enabled'] = $this->isBankTransferEnabled();
        
        return $status;
    }

    /**
     * Get available payment methods.
     * Only returns methods that are enabled AND properly configured.
     */
    public function getAvailableMethods(): array
    {
        $methods = [];
        $configStatus = $this->verifyConfiguration();
        
        if ($this->isStripeEnabled() && $configStatus['stripe']['configured']) {
            $methods[] = [
                'id' => 'stripe',
                'name' => 'Tarjeta de Crédito/Débito',
                'description' => 'Pago seguro con Stripe',
                'icon' => 'credit-card',
            ];
        } elseif ($this->isStripeEnabled() && !$configStatus['stripe']['configured']) {
            Log::warning('Stripe is enabled but not properly configured', [
                'error' => $configStatus['stripe']['error']
            ]);
        }
        
        if ($this->isPayPalEnabled() && $configStatus['paypal']['configured']) {
            $methods[] = [
                'id' => 'paypal',
                'name' => 'PayPal',
                'description' => 'Pago con cuenta PayPal',
                'icon' => 'paypal',
            ];
        } elseif ($this->isPayPalEnabled() && !$configStatus['paypal']['configured']) {
            Log::warning('PayPal is enabled but not properly configured', [
                'error' => $configStatus['paypal']['error']
            ]);
        }
        
        if ($this->isBankTransferEnabled()) {
            $methods[] = [
                'id' => 'bank_transfer',
                'name' => 'Transferencia Bancaria',
                'description' => 'Depósito o transferencia bancaria',
                'icon' => 'bank',
            ];
        }
        
        return $methods;
    }
    
    /**
     * Create a Stripe checkout session.
     */
    public function createStripeSession(Invoice $invoice, string $successUrl, string $cancelUrl): ?StripeSession
    {
        if (!$this->isStripeEnabled()) {
            throw new \Exception('Stripe no está habilitado.');
        }
        
        if (!$this->initStripe()) {
            throw new \Exception('Error al inicializar Stripe. Verifique la configuración de la API.');
        }
        
        $currency = strtolower(Setting::get('currency', 'USD'));
        $siteName = Setting::get('site_name', 'CoachPro');
        
        // Validate invoice total
        $total = (float) $invoice->total;
        if ($total <= 0) {
            throw new \Exception('El total de la factura debe ser mayor a 0.');
        }
        
        $amountInCents = (int) round($total * 100);
        
        Log::info('Stripe: Creating session', [
            'invoice_id' => $invoice->id,
            'total' => $total,
            'amount_cents' => $amountInCents,
            'currency' => $currency,
        ]);
        
        try {
            $session = StripeSession::create([
                'payment_method_types' => ['card'],
                'line_items' => [[
                    'price_data' => [
                        'currency' => $currency,
                        'product_data' => [
                            'name' => "Factura #{$invoice->invoice_number}",
                            'description' => $invoice->description ?? "Pago de suscripción - {$siteName}",
                        ],
                        'unit_amount' => $amountInCents,
                    ],
                    'quantity' => 1,
                ]],
                'mode' => 'payment',
                'success_url' => $successUrl . '?session_id={CHECKOUT_SESSION_ID}',
                'cancel_url' => $cancelUrl,
                'metadata' => [
                    'invoice_id' => $invoice->id,
                    'user_id' => $invoice->user_id,
                ],
                'customer_email' => $invoice->user->email,
            ]);
            
            Log::info('Stripe: Session created successfully', ['session_id' => $session->id]);
            
            return $session;
        } catch (ApiErrorException $e) {
            Log::error('Stripe API Error: ' . $e->getMessage(), [
                'code' => $e->getStripeCode(),
                'invoice_id' => $invoice->id,
            ]);
            throw new \Exception('Error de Stripe: ' . $e->getMessage());
        } catch (\Exception $e) {
            Log::error('Stripe Error: ' . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Verify Stripe payment was successful.
     */
    public function verifyStripePayment(string $sessionId): array
    {
        if (!$this->initStripe()) {
            throw new \Exception('Error al inicializar Stripe.');
        }
        
        try {
            $session = StripeSession::retrieve($sessionId);
            
            Log::info('Stripe: Payment verification', [
                'session_id' => $sessionId,
                'payment_status' => $session->payment_status,
            ]);
            
            return [
                'success' => $session->payment_status === 'paid',
                'session' => $session,
                'invoice_id' => $session->metadata->invoice_id ?? null,
            ];
        } catch (ApiErrorException $e) {
            Log::error('Stripe verification error: ' . $e->getMessage());
            throw new \Exception('Error al verificar el pago: ' . $e->getMessage());
        }
    }
    
    /**
     * Create a PayPal order.
     */
    public function createPayPalOrder(Invoice $invoice, string $successUrl, string $cancelUrl): array
    {
        if (!$this->isPayPalEnabled()) {
            throw new \Exception('PayPal no está habilitado.');
        }
        
        $paypal = $this->initPayPal();
        $currency = Setting::get('currency', 'USD');
        
        $order = $paypal->createOrder([
            'intent' => 'CAPTURE',
            'purchase_units' => [[
                'reference_id' => 'invoice_' . $invoice->id,
                'description' => "Factura #{$invoice->invoice_number}",
                'amount' => [
                    'currency_code' => $currency,
                    'value' => number_format($invoice->total, 2, '.', ''),
                ],
            ]],
            'application_context' => [
                'return_url' => $successUrl,
                'cancel_url' => $cancelUrl,
                'brand_name' => Setting::get('site_name', 'CoachPro'),
                'locale' => 'es-ES',
                'landing_page' => 'BILLING',
                'user_action' => 'PAY_NOW',
            ],
        ]);
        
        return $order;
    }
    
    /**
     * Capture PayPal payment.
     */
    public function capturePayPalPayment(string $orderId): array
    {
        $paypal = $this->initPayPal();
        
        return $paypal->capturePaymentOrder($orderId);
    }
    
    /**
     * Process bank transfer (mark as pending verification).
     */
    public function processBankTransfer(Invoice $invoice, array $proofData): bool
    {
        // Update invoice with proof of payment
        $invoice->update([
            'status' => 'pending_verification',
            'payment_proof' => $proofData['proof_image'] ?? null,
            'payment_reference' => $proofData['reference'] ?? null,
            'payment_date' => now(),
            'notes' => $proofData['notes'] ?? null,
        ]);
        
        return true;
    }
    
    /**
     * Mark invoice as paid.
     */
    public function markAsPaid(Invoice $invoice, string $method, ?string $transactionId = null): void
    {
        $invoice->update([
            'status' => 'paid',
            'payment_method' => $method,
            'transaction_id' => $transactionId,
            'paid_at' => now(),
        ]);
        
        // Activate subscription if applicable
        if ($invoice->subscription) {
            $invoice->subscription->activate();
        }
    }
}
