<?php

namespace App\Services;

use App\Models\Setting;
use Illuminate\Support\Facades\Log;

/**
 * Service to check notification settings before sending admin notifications.
 */
class NotificationSettingsService
{
    /**
     * Mapping of notification classes to their setting keys.
     */
    protected static array $notificationSettings = [
        // New user/client notifications
        'notify_new_user' => [
            \App\Notifications\NewCoachRegistered::class,
            \App\Notifications\AdminNewClientCreated::class,
        ],
        
        // Invoice/payment notifications
        'notify_invoice_paid' => [
            \App\Notifications\AdminPaymentReceived::class,
            \App\Notifications\AdminCoachPlanPurchased::class,
            \App\Notifications\AdminStorePaymentReceived::class,
            \App\Notifications\InvoicePaid::class,
        ],
        
        // Subscription/expiration notifications
        'notify_subscription_expired' => [
            \App\Notifications\AdminInvoiceOverdue::class,
            \App\Notifications\SubscriptionExpiringSoon::class,
            \App\Notifications\CoachPlanExpiringSoon::class,
            \App\Notifications\CoachPlanExpired::class,
        ],
        
        // Order notifications
        'notify_new_order' => [
            \App\Notifications\AdminNewStoreOrder::class,
            \App\Notifications\AdminOrderPaymentFailed::class,
            \App\Notifications\AdminOrderShipped::class,
            \App\Notifications\AdminOrderDelivered::class,
            \App\Notifications\OrderConfirmed::class,
        ],
    ];

    /**
     * Check if a specific notification type is enabled.
     */
    public static function isEnabled(string $settingKey): bool
    {
        try {
            return (bool) Setting::get($settingKey, true);
        } catch (\Exception $e) {
            Log::warning("NotificationSettings: Could not check setting {$settingKey}", [
                'error' => $e->getMessage(),
            ]);
            return true; // Default to enabled if setting check fails
        }
    }

    /**
     * Check if a notification class should be sent based on settings.
     */
    public static function shouldSend(string $notificationClass): bool
    {
        foreach (self::$notificationSettings as $settingKey => $classes) {
            if (in_array($notificationClass, $classes)) {
                return self::isEnabled($settingKey);
            }
        }
        
        // If notification not mapped, allow it by default
        return true;
    }

    /**
     * Get the setting key for a notification class.
     */
    public static function getSettingKey(string $notificationClass): ?string
    {
        foreach (self::$notificationSettings as $settingKey => $classes) {
            if (in_array($notificationClass, $classes)) {
                return $settingKey;
            }
        }
        return null;
    }

    /**
     * Get all notification settings status.
     */
    public static function getAllSettings(): array
    {
        return [
            'notify_new_user' => self::isEnabled('notify_new_user'),
            'notify_invoice_paid' => self::isEnabled('notify_invoice_paid'),
            'notify_subscription_expired' => self::isEnabled('notify_subscription_expired'),
            'notify_new_order' => self::isEnabled('notify_new_order'),
        ];
    }
}
