<?php

namespace App\Notifications;

use App\Models\Setting;
use App\Models\WalletTransaction;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class WalletDepositConfirmed extends Notification implements ShouldQueue
{
    use Queueable;

    public function __construct(
        public WalletTransaction $transaction
    ) {}

    public function via(object $notifiable): array
    {
        return ['database', 'mail'];
    }

    public function toMail(object $notifiable): MailMessage
    {
        $siteName = Setting::get('site_name', 'CoachPro');
        $currency = Setting::get('currency', 'DOP');
        $amount = number_format(abs($this->transaction->amount), 2);

        return (new MailMessage)
            ->subject("Depósito Confirmado - \${$amount} {$currency}")
            ->greeting("¡Hola {$notifiable->name}!")
            ->line("Tu depósito ha sido confirmado y acreditado a tu billetera.")
            ->line("**Monto:** \${$amount} {$currency}")
            ->line("**Método:** " . $this->getMethodName())
            ->line("**Referencia:** " . ($this->transaction->payment_reference ?? 'N/A'))
            ->action('Ver Mi Billetera', route('coach.wallet.index'))
            ->line("Tu nuevo saldo está disponible para uso inmediato.")
            ->salutation("Saludos,\nEl equipo de {$siteName}");
    }

    public function toArray(object $notifiable): array
    {
        $amount = number_format(abs($this->transaction->amount), 2);

        return [
            'type' => 'wallet_deposit_confirmed',
            'transaction_id' => $this->transaction->id,
            'amount' => abs($this->transaction->amount),
            'payment_method' => $this->transaction->payment_method,
            'message' => "✅ Depósito confirmado: \${$amount} acreditado a tu billetera",
            'title' => 'Depósito confirmado',
            'url' => route('coach.wallet.index'),
        ];
    }

    protected function getMethodName(): string
    {
        return match($this->transaction->payment_method) {
            'stripe' => 'Tarjeta (Stripe)',
            'paypal' => 'PayPal',
            'bank_transfer' => 'Transferencia Bancaria',
            default => ucfirst($this->transaction->payment_method ?? 'Otro'),
        };
    }
}
