<?php

namespace App\Notifications;

use App\Models\Setting;
use App\Models\User;
use App\Models\Workout;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class AdminWorkoutCreated extends Notification implements ShouldQueue
{
    use Queueable;

    /**
     * Create a new notification instance.
     */
    public function __construct(
        public Workout $workout,
        public User $coach
    ) {}

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['database', 'mail'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $siteName = Setting::get('site_name', 'CoachPro');
        $clientName = $this->workout->client?->name ?? 'Cliente';

        return (new MailMessage)
            ->subject("🏋️ Nueva Rutina Creada - {$this->coach->name}")
            ->greeting('¡Hola Administrador!')
            ->line("El coach {$this->coach->name} ha creado una nueva rutina de entrenamiento.")
            ->line('')
            ->line('**Detalles de la Rutina:**')
            ->line("Nombre: {$this->workout->name}")
            ->line("Cliente: {$clientName}")
            ->line("Coach: {$this->coach->name}")
            ->line("Fecha: " . now()->format('d/m/Y H:i'))
            ->action('Ver Coach', route('admin.users.show', $this->coach->id))
            ->line('La rutina ya está disponible para el cliente.')
            ->salutation("Saludos, {$siteName}");
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            'type' => 'admin_workout_created',
            'workout_id' => $this->workout->id,
            'workout_name' => $this->workout->name,
            'client_name' => $this->workout->client?->name ?? 'Cliente',
            'coach_id' => $this->coach->id,
            'coach_name' => $this->coach->name,
            'message' => "🏋️ Nueva rutina: {$this->workout->name} creada por {$this->coach->name}",
            'title' => 'Nueva rutina de entrenamiento',
            'url' => route('admin.users.show', $this->coach->id),
        ];
    }
}
