<?php

namespace App\Notifications;

use App\Models\Order;
use App\Models\User;
use App\Notifications\Traits\ChecksNotificationSettings;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

/**
 * Notification sent to administrators when a store payment is received.
 */
class AdminStorePaymentReceived extends Notification implements ShouldQueue
{
    use Queueable, ChecksNotificationSettings;

    public function __construct(
        public Order $order,
        public ?string $transactionId = null
    ) {}

    public function via(object $notifiable): array
    {
        // Database first to ensure in-app notification is saved even if email fails
        return ['database', 'mail'];
    }

    public function toMail(object $notifiable): MailMessage
    {
        $message = (new MailMessage)
            ->subject('💰 Pago Recibido - Orden #' . $this->order->order_number)
            ->greeting('¡Pago confirmado!')
            ->line('Se ha recibido el pago de un pedido de la tienda.')
            ->line('')
            ->line('**Detalles del Pago:**')
            ->line('**Orden:** #' . $this->order->order_number)
            ->line('**Monto:** ' . $this->order->formatCurrency($this->order->total))
            ->line('**Método:** ' . $this->getPaymentMethodLabel())
            ->line('**Fecha:** ' . now()->format('d/m/Y H:i'))
            ->line('')
            ->line('**Cliente:**')
            ->line('**Nombre:** ' . $this->order->user->name)
            ->line('**Email:** ' . $this->order->user->email);

        if ($this->transactionId) {
            $message->line('')
                ->line('**ID de Transacción:** ' . $this->transactionId);
        }

        return $message
            ->line('')
            ->line('El pedido está listo para ser procesado.')
            ->action('Procesar Pedido', route('admin.store.orders.show', $this->order->id))
            ->salutation('Sistema de Notificaciones CoachPro');
    }

    public function toArray(object $notifiable): array
    {
        return [
            'type' => 'admin_store_payment_received',
            'order_id' => $this->order->id,
            'order_number' => $this->order->order_number,
            'customer_name' => $this->order->user->name,
            'amount' => $this->order->total,
            'payment_method' => $this->order->payment_method,
            'transaction_id' => $this->transactionId,
            'message' => "Pago de " . $this->order->formatCurrency($this->order->total) . " recibido para orden #{$this->order->order_number}",
            'title' => 'Pago Recibido (Tienda)',
            'url' => route('admin.store.orders.show', $this->order->id),
            'icon' => 'credit-card',
            'priority' => 'high',
        ];
    }

    protected function getPaymentMethodLabel(): string
    {
        $labels = [
            'paypal' => 'PayPal',
            'stripe' => 'Tarjeta (Stripe)',
            'card' => 'Tarjeta de Crédito',
            'transfer' => 'Transferencia Bancaria',
        ];

        return $labels[$this->order->payment_method] ?? $this->order->payment_method;
    }

    public function databaseType(object $notifiable): string
    {
        return 'admin_store_payment_received';
    }
}
