<?php

namespace App\Notifications;

use App\Models\Order;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

/**
 * Notification sent to administrators when an order is delivered.
 */
class AdminOrderDelivered extends Notification implements ShouldQueue
{
    use Queueable;

    public function __construct(
        public Order $order
    ) {}

    public function via(object $notifiable): array
    {
        // Database first to ensure in-app notification is saved even if email fails
        return ['database', 'mail'];
    }

    public function toMail(object $notifiable): MailMessage
    {
        return (new MailMessage)
            ->subject('✅ Pedido Entregado - #' . $this->order->order_number)
            ->greeting('Pedido entregado')
            ->line('Un pedido ha sido entregado exitosamente.')
            ->line('')
            ->line('**Detalles:**')
            ->line('**Orden:** #' . $this->order->order_number)
            ->line('**Cliente:** ' . $this->order->user->name)
            ->line('**Email:** ' . $this->order->user->email)
            ->line('**Total:** ' . $this->order->formatCurrency($this->order->total))
            ->line('**Fecha de Entrega:** ' . $this->order->delivered_at?->format('d/m/Y H:i'))
            ->line('')
            ->action('Ver Pedido', route('admin.store.orders.show', $this->order->id))
            ->salutation('Sistema de Notificaciones CoachPro');
    }

    public function toArray(object $notifiable): array
    {
        return [
            'type' => 'admin_order_delivered',
            'order_id' => $this->order->id,
            'order_number' => $this->order->order_number,
            'customer_name' => $this->order->user->name,
            'total' => $this->order->total,
            'delivered_at' => $this->order->delivered_at?->toISOString(),
            'message' => "Pedido #{$this->order->order_number} entregado a {$this->order->user->name}",
            'title' => 'Pedido Entregado',
            'url' => route('admin.store.orders.show', $this->order->id),
            'icon' => 'check-circle',
            'priority' => 'normal',
        ];
    }

    public function databaseType(object $notifiable): string
    {
        return 'admin_order_delivered';
    }
}
