<?php

namespace App\Notifications;

use App\Models\Invoice;
use App\Models\Setting;
use App\Notifications\Traits\ChecksNotificationSettings;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class AdminInvoiceOverdue extends Notification implements ShouldQueue
{
    use Queueable, ChecksNotificationSettings;

    /**
     * Create a new notification instance.
     */
    public function __construct(
        public Invoice $invoice,
        public int $daysPastDue = 0
    ) {}

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['database', 'mail'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $user = $this->invoice->user;
        $siteName = Setting::get('site_name', 'CoachPro');

        return (new MailMessage)
            ->subject("⚠️ Factura Vencida - {$user->name}")
            ->greeting("¡Hola Administrador!")
            ->line("La siguiente factura está vencida y requiere atención:")
            ->line("")
            ->line("**Detalles de la Factura:**")
            ->line("Número: {$this->invoice->invoice_number}")
            ->line("Coach: {$user->name} ({$user->email})")
            ->line("Total: $" . number_format($this->invoice->total, 2))
            ->line("Días vencidos: {$this->daysPastDue}")
            ->action('Ver Factura', route('admin.invoices.show', $this->invoice->id))
            ->line("Por favor contacta al coach para resolver este pago pendiente.")
            ->salutation("Saludos, {$siteName}");
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        $user = $this->invoice->user;

        return [
            'type' => 'admin_invoice_overdue',
            'invoice_id' => $this->invoice->id,
            'invoice_number' => $this->invoice->invoice_number,
            'user_id' => $user->id,
            'user_name' => $user->name,
            'total' => $this->invoice->total,
            'days_past_due' => $this->daysPastDue,
            'message' => "⚠️ Factura vencida: {$user->name} debe \${$this->invoice->total} ({$this->daysPastDue} días)",
            'title' => 'Factura vencida de coach',
            'url' => route('admin.users.show', $user->id),
        ];
    }
}
