<?php

namespace App\Notifications;

use App\Models\Plan;
use App\Models\Setting;
use App\Models\Subscription;
use App\Models\User;
use App\Notifications\Traits\ChecksNotificationSettings;
use App\Notifications\Traits\LogsEmailNotification;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

/**
 * Notificación detallada al administrador cuando un coach compra un plan.
 * 
 * Incluye:
 * - Datos completos del coach
 * - Detalles del plan y precio
 * - Método de pago utilizado
 * - Información de facturación
 */
class AdminCoachSubscriptionPurchased extends Notification implements ShouldQueue
{
    use Queueable, ChecksNotificationSettings, LogsEmailNotification;

    public function __construct(
        public User $coach,
        public Subscription $subscription,
        public ?string $paymentMethod = null,
        public ?float $amount = null
    ) {}

    public function via(object $notifiable): array
    {
        // Database first to ensure in-app notification is saved even if mail fails
        return ['database', 'mail'];
    }

    public function toMail(object $notifiable): MailMessage
    {
        $appName = Setting::get('app_name', 'CoachPro');
        $planName = $this->subscription->plan->name ?? 'N/A';
        $subject = "💰 Nueva Suscripción: {$this->coach->name} - Plan {$planName}";

        // Create email log
        $this->createEmailLog($notifiable, $subject, $this->getEmailLogMetadata());

        $amount = $this->amount ?? $this->subscription->plan->price ?? 0;
        $paymentMethod = $this->paymentMethod ?? 'No especificado';
        
        $endDate = $this->subscription->end_date 
            ? $this->subscription->end_date->format('d/m/Y')
            : 'N/A';

        $mail = (new MailMessage)
            ->subject($subject)
            ->greeting('¡Nueva suscripción recibida!')
            ->line("Un coach ha adquirido un nuevo plan en {$appName}.")
            ->line('')
            ->line('---')
            ->line('## 👤 Información del Coach')
            ->line('')
            ->line("**Nombre:** {$this->coach->name}")
            ->line("**Email:** {$this->coach->email}")
            ->line("**ID:** #{$this->coach->id}")
            ->line("**Fecha de registro:** {$this->coach->created_at->format('d/m/Y H:i')}")
            ->line('')
            ->line('---')
            ->line('## 📋 Detalles de la Suscripción')
            ->line('')
            ->line("**Plan:** {$planName}")
            ->line("**Precio:** \${$amount}")
            ->line("**Método de pago:** {$paymentMethod}")
            ->line("**Estado:** {$this->subscription->status}")
            ->line("**Válida hasta:** {$endDate}")
            ->line('')
            ->line('---')
            ->line('## 📊 Estadísticas')
            ->line('')
            ->line("**Total coaches activos:** " . User::where('role', 'coach')->where('status', 'active')->count())
            ->line("**Suscripciones activas:** " . Subscription::where('status', 'active')->count())
            ->line('')
            ->action('Ver Detalles del Coach', route('admin.users.show', $this->coach->id))
            ->line('')
            ->salutation("Sistema de Notificaciones de {$appName}");

        return $mail;
    }

    protected function getEmailLogMetadata(): array
    {
        return [
            'coach_id' => $this->coach->id,
            'coach_name' => $this->coach->name,
            'subscription_id' => $this->subscription->id,
            'plan_id' => $this->subscription->plan_id,
            'plan_name' => $this->subscription->plan->name ?? 'N/A',
            'amount' => $this->amount ?? $this->subscription->plan->price ?? 0,
            'payment_method' => $this->paymentMethod,
        ];
    }

    public function toArray(object $notifiable): array
    {
        $planName = $this->subscription->plan->name ?? 'N/A';
        $amount = $this->amount ?? $this->subscription->plan->price ?? 0;

        return [
            'type' => 'admin_coach_subscription',
            'coach_id' => $this->coach->id,
            'coach_name' => $this->coach->name,
            'subscription_id' => $this->subscription->id,
            'plan_name' => $planName,
            'amount' => $amount,
            'message' => "💰 {$this->coach->name} adquirió el plan {$planName} por \${$amount}",
            'title' => 'Nueva suscripción',
            'url' => route('admin.subscriptions.show', $this->subscription->id),
        ];
    }
}
