<?php

namespace App\Notifications;

use App\Models\CoachPlanPurchase;
use App\Models\Setting;
use App\Notifications\Traits\ChecksNotificationSettings;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class AdminCoachPlanPurchased extends Notification implements ShouldQueue
{
    use Queueable, ChecksNotificationSettings;

    /**
     * Create a new notification instance.
     */
    public function __construct(
        public CoachPlanPurchase $purchase
    ) {}

    /**
     * Get the notification's delivery channels.
     * Database first to ensure in-app notification is saved even if email fails.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['database', 'mail'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $siteName = Setting::get('site_name', 'CoachPro');
        $currency = Setting::get('currency', 'USD');
        $plan = $this->purchase->coachPlan;
        $client = $this->purchase->client;
        $coach = $plan?->coach;
        $clientName = $client?->name ?? $this->purchase->buyer_name ?? 'Cliente desconocido';
        $coachName = $coach?->name ?? 'Coach';
        $coachEmail = $coach?->email ?? '';

        return (new MailMessage)
            ->subject("💰 Venta de Plan de Coach - " . ($plan?->name ?? 'Plan'))
            ->greeting("¡Nueva venta de plan de coach!")
            ->line("Un cliente ha comprado un plan personalizado de un coach.")
            ->line('')
            ->line('**Detalles de la Venta:**')
            ->line("Coach: {$coachName}" . ($coachEmail ? " ({$coachEmail})" : ''))
            ->line("Cliente: {$clientName}")
            ->line("Plan: " . ($plan?->name ?? 'N/A'))
            ->line("Monto: \${$this->purchase->amount} {$currency}")
            ->line("Método de Pago: " . $this->getPaymentMethodName())
            ->action('Ver Panel de Admin', route('admin.dashboard'))
            ->salutation("Sistema de Notificaciones - {$siteName}");
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        $plan = $this->purchase->coachPlan;
        $client = $this->purchase->client;
        $coach = $plan?->coach;
        $clientName = $client?->name ?? $this->purchase->buyer_name ?? 'Cliente desconocido';
        $coachName = $coach?->name ?? 'Coach';
        $planName = $plan?->name ?? 'Plan';

        return [
            'type' => 'coach_plan_purchased',
            'purchase_id' => $this->purchase->id,
            'plan_name' => $planName,
            'coach_id' => $coach?->id,
            'coach_name' => $coachName,
            'client_name' => $clientName,
            'amount' => $this->purchase->amount,
            'payment_method' => $this->purchase->payment_method,
            'message' => "💰 Venta de plan: {$clientName} compró '{$planName}' de {$coachName} por \${$this->purchase->amount}",
            'title' => 'Venta de plan de coach',
            'url' => $coach ? route('admin.users.show', $coach->id) : '#',
        ];
    }

    /**
     * Get human-readable payment method name.
     */
    protected function getPaymentMethodName(): string
    {
        return match($this->purchase->payment_method) {
            'paypal' => 'PayPal',
            'paypal_me' => 'PayPal.me',
            'transfer' => 'Transferencia',
            'cash' => 'Efectivo',
            'other' => 'Otro',
            default => ucfirst($this->purchase->payment_method ?? 'Desconocido'),
        };
    }
}
