<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Facades\DB;

class Wallet extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'balance',
        'currency',
    ];

    protected $casts = [
        'balance' => 'decimal:2',
    ];

    /**
     * Get the user that owns the wallet.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the transactions for the wallet.
     */
    public function transactions(): HasMany
    {
        return $this->hasMany(WalletTransaction::class)->latest();
    }

    /**
     * Add funds to the wallet.
     */
    public function deposit(float $amount, string $description = null, array $metadata = []): WalletTransaction
    {
        return DB::transaction(function () use ($amount, $description, $metadata) {
            $this->increment('balance', $amount);
            $this->refresh();

            return $this->transactions()->create([
                'type' => 'deposit',
                'amount' => $amount,
                'balance_after' => $this->balance,
                'description' => $description ?? 'Depósito',
                'status' => 'completed',
                'metadata' => $metadata,
            ]);
        });
    }

    /**
     * Withdraw funds from the wallet.
     */
    public function withdraw(float $amount, string $description = null, array $metadata = []): WalletTransaction
    {
        if ($this->balance < $amount) {
            throw new \Exception('Saldo insuficiente');
        }

        return DB::transaction(function () use ($amount, $description, $metadata) {
            $this->decrement('balance', $amount);
            $this->refresh();

            return $this->transactions()->create([
                'type' => 'withdrawal',
                'amount' => -$amount,
                'balance_after' => $this->balance,
                'description' => $description ?? 'Retiro',
                'status' => 'pending', // Withdrawals need admin approval
                'metadata' => $metadata,
            ]);
        });
    }

    /**
     * Add referral bonus.
     */
    public function addReferralBonus(float $amount, int $referredUserId, string $description = null): WalletTransaction
    {
        return DB::transaction(function () use ($amount, $referredUserId, $description) {
            $this->increment('balance', $amount);
            $this->refresh();

            return $this->transactions()->create([
                'type' => 'referral_bonus',
                'amount' => $amount,
                'balance_after' => $this->balance,
                'description' => $description ?? 'Bono por referido',
                'reference_type' => 'user',
                'reference_id' => $referredUserId,
                'status' => 'completed',
            ]);
        });
    }

    /**
     * Create a pending deposit transaction (for bank transfers awaiting confirmation).
     */
    public function createPendingDeposit(float $amount, string $paymentMethod, string $paymentReference = null, array $metadata = []): WalletTransaction
    {
        return $this->transactions()->create([
            'type' => 'deposit',
            'amount' => $amount,
            'balance_after' => null, // Will be set when confirmed
            'description' => 'Recarga de saldo - ' . ucfirst($paymentMethod),
            'payment_method' => $paymentMethod,
            'payment_reference' => $paymentReference,
            'status' => 'pending',
            'metadata' => $metadata,
        ]);
    }

    /**
     * Confirm a pending transaction.
     */
    public function confirmTransaction(WalletTransaction $transaction): void
    {
        if ($transaction->status !== 'pending') {
            throw new \Exception('La transacción no está pendiente');
        }

        DB::transaction(function () use ($transaction) {
            if ($transaction->type === 'deposit') {
                $this->increment('balance', abs($transaction->amount));
            }
            
            $this->refresh();
            $transaction->update([
                'status' => 'completed',
                'balance_after' => $this->balance,
            ]);
        });
    }

    /**
     * Pay for a subscription using wallet balance.
     */
    public function paySubscription(float $amount, int $subscriptionId, string $planName): WalletTransaction
    {
        if ($this->balance < $amount) {
            throw new \Exception('Saldo insuficiente en la billetera');
        }

        return DB::transaction(function () use ($amount, $subscriptionId, $planName) {
            $this->decrement('balance', $amount);
            $this->refresh();

            return $this->transactions()->create([
                'type' => 'subscription_payment',
                'amount' => -$amount,
                'balance_after' => $this->balance,
                'description' => "Pago de suscripción - Plan {$planName}",
                'reference_type' => 'subscription',
                'reference_id' => $subscriptionId,
                'status' => 'completed',
                'metadata' => [
                    'subscription_id' => $subscriptionId,
                    'plan_name' => $planName,
                    'paid_at' => now()->toISOString(),
                ],
            ]);
        });
    }

    /**
     * Check if wallet has sufficient balance.
     */
    public function hasSufficientBalance(float $amount): bool
    {
        return $this->balance >= $amount;
    }
}
