<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Plan extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'slug',
        'price',
        'monthly_price',
        'quarterly_price',
        'yearly_price',
        'duration_days',
        'is_active',
        'is_featured',
        'features',
        'description',
        'paypal_plan_id',
        'stripe_price_id',
    ];

    protected $casts = [
        'price' => 'decimal:2',
        'monthly_price' => 'decimal:2',
        'quarterly_price' => 'decimal:2',
        'yearly_price' => 'decimal:2',
        'is_active' => 'boolean',
        'is_featured' => 'boolean',
        'features' => 'array',
    ];

    /**
     * Get subscriptions for this plan.
     */
    public function subscriptions(): HasMany
    {
        return $this->hasMany(Subscription::class);
    }

    /**
     * Check if a feature is available.
     */
    public function hasFeature(string $feature): bool
    {
        return isset($this->features[$feature]) && $this->features[$feature];
    }

    /**
     * Get a feature limit (-1 = unlimited).
     */
    public function getLimit(string $feature): int
    {
        return $this->features[$feature] ?? 0;
    }

    /**
     * Check if a limit is unlimited.
     */
    public function isUnlimited(string $feature): bool
    {
        return $this->getLimit($feature) === -1;
    }

    /**
     * Get price for billing cycle.
     */
    public function getPriceForCycle(string $cycle): float
    {
        return match($cycle) {
            'monthly' => $this->monthly_price ?? $this->price,
            'quarterly' => $this->quarterly_price ?? ($this->price * 3 * 0.9), // 10% discount
            'yearly' => $this->yearly_price ?? ($this->price * 12 * 0.8), // 20% discount
            default => $this->price,
        };
    }

    /**
     * Check if plan supports recurring billing.
     */
    public function supportsRecurring(): bool
    {
        return $this->monthly_price || $this->paypal_plan_id || $this->stripe_price_id;
    }

    /**
     * Get the route key for the model.
     */
    public function getRouteKeyName(): string
    {
        return 'slug';
    }
}
