<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class Food extends Model
{
    use HasFactory;

    /**
     * The table associated with the model.
     * 
     * Laravel doesn't automatically pluralize "food" to "foods"
     */
    protected $table = 'foods';

    protected $fillable = [
        'user_id',
        'name',
        'category',
        'meal_type',
        'serving_size',
        'serving_weight',
        'proteins',
        'carbs',
        'fats',
        'calories',
        'fiber',
        'is_custom',
        'is_active',
    ];

    protected $casts = [
        'proteins' => 'decimal:2',
        'carbs' => 'decimal:2',
        'fats' => 'decimal:2',
        'calories' => 'decimal:2',
        'fiber' => 'decimal:2',
        'serving_weight' => 'decimal:2',
        'is_custom' => 'boolean',
        'is_active' => 'boolean',
    ];

    /**
     * Tipos de comida disponibles
     */
    public const MEAL_TYPES = [
        'desayuno' => 'Desayuno',
        'almuerzo' => 'Almuerzo',
        'merienda' => 'Merienda',
        'merienda_media_manana' => 'Merienda Media Mañana',
        'merienda_media_tarde' => 'Merienda Media Tarde',
        'cena' => 'Cena',
        'pre_workout' => 'Pre-Entreno',
        'intra_workout' => 'Intra-Entreno',
        'post_workout' => 'Post-Entreno',
        'comida_1' => 'Comida 1',
        'comida_2' => 'Comida 2',
        'comida_3' => 'Comida 3',
        'comida_4' => 'Comida 4',
        'comida_5' => 'Comida 5',
        'comida_6' => 'Comida 6',
        'comida_7' => 'Comida 7',
        'comida_8' => 'Comida 8',
    ];

    /**
     * Get the coach who created this custom food.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Calculate macros for a given quantity in grams.
     */
    public function calculateMacros(float $quantity): array
    {
        $factor = $quantity / 100;

        return [
            'proteins' => round($this->proteins * $factor, 2),
            'carbs' => round($this->carbs * $factor, 2),
            'fats' => round($this->fats * $factor, 2),
            'calories' => round($this->calories * $factor, 2),
            'fiber' => round($this->fiber * $factor, 2),
        ];
    }

    /**
     * Scope for global foods (not custom).
     */
    public function scopeGlobal($query)
    {
        return $query->where('is_custom', false);
    }

    /**
     * Scope for custom foods of a specific coach.
     */
    public function scopeForCoach($query, int $coachId)
    {
        return $query->where('is_custom', true)
            ->where('user_id', $coachId);
    }

    /**
     * Scope for available foods (global + coach's custom).
     */
    public function scopeAvailableFor($query, int $coachId)
    {
        return $query->where(function ($q) use ($coachId) {
            $q->where('is_custom', false)
                ->orWhere(function ($q2) use ($coachId) {
                    $q2->where('is_custom', true)
                        ->where('user_id', $coachId);
                });
        });
    }
}
