<?php

namespace App\Http\Traits;

use Illuminate\Http\Request;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;

/**
 * Trait for handling file uploads in controllers.
 * 
 * Provides methods to upload, update and delete files from storage.
 * 
 * @example
 * use App\Http\Traits\HandlesFileUploads;
 * 
 * class ExampleController extends Controller
 * {
 *     use HandlesFileUploads;
 * 
 *     public function store(Request $request)
 *     {
 *         $data = $request->validated();
 *         $data['image'] = $this->uploadFile($request, 'image', 'examples');
 *     }
 * }
 */
trait HandlesFileUploads
{
    /**
     * Upload a file from a request.
     *
     * @param Request $request The request containing the file
     * @param string $field The form field name
     * @param string $folder The storage folder
     * @param string $disk The storage disk (default: 'public')
     * @return string|null The stored file path or null if no file
     */
    protected function uploadFile(Request $request, string $field, string $folder, string $disk = 'public'): ?string
    {
        if (!$request->hasFile($field)) {
            return null;
        }

        return $request->file($field)->store($folder, $disk);
    }

    /**
     * Upload a file and delete the old one if exists.
     *
     * @param Request $request The request containing the file
     * @param string $field The form field name
     * @param string $folder The storage folder
     * @param string|null $oldPath The old file path to delete
     * @param string $disk The storage disk (default: 'public')
     * @return string|null The new file path, old path if no new file, or null
     */
    protected function updateFile(Request $request, string $field, string $folder, ?string $oldPath = null, string $disk = 'public'): ?string
    {
        if (!$request->hasFile($field)) {
            return $oldPath;
        }

        // Delete old file if exists
        $this->deleteFile($oldPath, $disk);

        // Store new file
        return $request->file($field)->store($folder, $disk);
    }

    /**
     * Delete a file from storage.
     *
     * @param string|null $path The file path to delete
     * @param string $disk The storage disk (default: 'public')
     * @return bool True if deleted, false otherwise
     */
    protected function deleteFile(?string $path, string $disk = 'public'): bool
    {
        if (!$path) {
            return false;
        }

        if (Storage::disk($disk)->exists($path)) {
            return Storage::disk($disk)->delete($path);
        }

        return false;
    }

    /**
     * Upload multiple files from a request.
     *
     * @param Request $request The request containing the files
     * @param string $field The form field name
     * @param string $folder The storage folder
     * @param string $disk The storage disk (default: 'public')
     * @return array The array of stored file paths
     */
    protected function uploadMultipleFiles(Request $request, string $field, string $folder, string $disk = 'public'): array
    {
        if (!$request->hasFile($field)) {
            return [];
        }

        $files = $request->file($field);
        $paths = [];

        foreach ($files as $file) {
            if ($file instanceof UploadedFile && $file->isValid()) {
                $paths[] = $file->store($folder, $disk);
            }
        }

        return $paths;
    }

    /**
     * Delete multiple files from storage.
     *
     * @param array $paths The file paths to delete
     * @param string $disk The storage disk (default: 'public')
     * @return int Number of files deleted
     */
    protected function deleteMultipleFiles(array $paths, string $disk = 'public'): int
    {
        $deleted = 0;

        foreach ($paths as $path) {
            if ($this->deleteFile($path, $disk)) {
                $deleted++;
            }
        }

        return $deleted;
    }

    /**
     * Get the full URL for a stored file.
     *
     * @param string|null $path The file path
     * @param string $disk The storage disk (default: 'public')
     * @return string|null The full URL or null
     */
    protected function getFileUrl(?string $path, string $disk = 'public'): ?string
    {
        if (!$path) {
            return null;
        }

        return Storage::disk($disk)->url($path);
    }
}
