<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

class EnsureActiveSubscription
{
    /**
     * Routes that should be accessible without active subscription.
     * Allows coaches to access subscription management pages.
     */
    protected array $except = [
        'coach/subscription',
        'coach/subscription/*',
        'coach/subscription/*/*',
        'coach/subscription/*/*/*',
        'coach/subscription/*/*/*/*',
        'coach/payment/*',
        'coach/payment/*/*',
        'coach/payment/*/*/*',
    ];

    /**
     * Handle an incoming request.
     * Ensures the coach has an active subscription before accessing protected routes.
     */
    public function handle(Request $request, Closure $next): Response
    {
        $user = $request->user();

        if (!$user) {
            return redirect()->route('login');
        }

        // Only check subscription for coaches
        if ($user->isCoach() && !$user->hasActiveSubscription()) {
            // Allow access to subscription-related routes
            if ($this->shouldPassThrough($request)) {
                return $next($request);
            }

            return redirect()->route('subscription.expired')
                ->with('warning', 'Tu suscripción ha expirado. Por favor, renueva para continuar.');
        }

        return $next($request);
    }

    /**
     * Determine if the request should pass through the middleware.
     */
    protected function shouldPassThrough(Request $request): bool
    {
        foreach ($this->except as $pattern) {
            if ($request->is($pattern)) {
                return true;
            }
        }

        return false;
    }
}
