<?php

namespace App\Http\Controllers;

use App\Models\Order;
use App\Models\Setting;
use App\Models\User;
use App\Notifications\OrderPaid;
use App\Notifications\OrderPaymentFailed;
use App\Notifications\AdminStorePaymentReceived;
use App\Notifications\AdminOrderPaymentFailed;
use App\Services\PaymentService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Notification;
use Inertia\Inertia;
use Srmklive\PayPal\Services\PayPal as PayPalClient;
use Stripe\Stripe;
use Stripe\Checkout\Session as StripeSession;

class StorePaymentController extends Controller
{
    protected PaymentService $paymentService;

    public function __construct(PaymentService $paymentService)
    {
        $this->paymentService = $paymentService;
    }

    /**
     * Initiate PayPal payment for store order.
     */
    public function paypalCreate(Order $order)
    {
        // Verify order belongs to current user
        /** @var int|null $userId */
        $userId = auth()->id();
        if ($order->user_id !== $userId) {
            abort(403);
        }

        // Verify order is awaiting payment
        if (!in_array($order->status, ['awaiting_payment', 'pending'])) {
            return redirect()->route('client.store.orders.show', $order->id)
                ->with('error', 'Este pedido ya ha sido procesado.');
        }

        try {
            $paypal = $this->initPayPal();

            $paypalOrder = $paypal->createOrder([
                'intent' => 'CAPTURE',
                'purchase_units' => [
                    [
                        'reference_id' => 'order_' . $order->id,
                        'description' => 'Pedido #' . $order->order_number,
                        'amount' => [
                            'currency_code' => 'DOP',
                            'value' => number_format($order->total, 2, '.', ''),
                        ],
                    ],
                ],
                'application_context' => [
                    'return_url' => route('store.payment.paypal.success', ['order' => $order->id]),
                    'cancel_url' => route('store.payment.paypal.cancel', ['order' => $order->id]),
                    'brand_name' => config('app.name'),
                    'user_action' => 'PAY_NOW',
                ],
            ]);

            if (isset($paypalOrder['id'])) {
                // Store PayPal order ID
                $order->update(['payment_id' => $paypalOrder['id']]);

                // Find approval link
                foreach ($paypalOrder['links'] as $link) {
                    if ($link['rel'] === 'approve') {
                        return Inertia::location($link['href']);
                    }
                }
            }

            Log::error('PayPal order creation failed', ['response' => $paypalOrder]);
            return redirect()->route('client.store.orders.show', $order->id)
                ->with('error', 'Error al procesar el pago con PayPal.');

        } catch (\Exception $e) {
            Log::error('PayPal error: ' . $e->getMessage());
            return redirect()->route('client.store.orders.show', $order->id)
                ->with('error', 'Error al conectar con PayPal: ' . $e->getMessage());
        }
    }

    /**
     * PayPal payment success callback.
     */
    public function paypalSuccess(Request $request, Order $order)
    {
        /** @var int|null $userId */
        $userId = auth()->id();
        if ($order->user_id !== $userId) {
            abort(403);
        }

        $paypalOrderId = $request->query('token');

        try {
            $paypal = $this->initPayPal();
            $result = $paypal->capturePaymentOrder($paypalOrderId);

            if (isset($result['status']) && $result['status'] === 'COMPLETED') {
                $order->update([
                    'status' => 'processing',
                    'payment_status' => 'paid',
                    'paid_at' => now(),
                    'transaction_id' => $paypalOrderId,
                ]);

                // Send payment confirmation to customer
                $order->user->notify(new OrderPaid($order));

                // Notify admins about payment received
                $admins = User::where('role', 'admin')->get();
                Notification::send($admins, new AdminStorePaymentReceived($order, $paypalOrderId));

                return redirect()->route('client.store.orders.show', $order->id)
                    ->with('success', '¡Pago completado! Tu pedido está siendo procesado.');
            }

            Log::error('PayPal capture failed', ['result' => $result]);

            // Notify user and admins about failed payment
            $this->notifyPaymentFailed($order, 'El pago con PayPal no pudo ser completado', 'paypal');

            return redirect()->route('client.store.orders.show', $order->id)
                ->with('error', 'No se pudo completar el pago.');

        } catch (\Exception $e) {
            Log::error('PayPal capture error: ' . $e->getMessage());

            // Notify user and admins about failed payment
            $this->notifyPaymentFailed($order, $e->getMessage(), 'paypal');

            return redirect()->route('client.store.orders.show', $order->id)
                ->with('error', 'Error al verificar el pago.');
        }
    }

    /**
     * PayPal payment cancelled.
     */
    public function paypalCancel(Order $order)
    {
        return redirect()->route('client.store.orders.show', $order->id)
            ->with('warning', 'Pago cancelado. Puedes intentar de nuevo cuando quieras.');
    }

    /**
     * Initiate Stripe payment for store order.
     */
    public function stripeCreate(Order $order)
    {
        /** @var int|null $userId */
        $userId = auth()->id();
        if ($order->user_id !== $userId) {
            abort(403);
        }

        if (!in_array($order->status, ['awaiting_payment', 'pending'])) {
            return redirect()->route('client.store.orders.show', $order->id)
                ->with('error', 'Este pedido ya ha sido procesado.');
        }

        try {
            $stripeSecret = Setting::get('stripe_secret');
            if (!$stripeSecret) {
                throw new \Exception('Stripe no está configurado.');
            }

            // Try to decrypt
            try {
                $stripeSecret = decrypt($stripeSecret);
            } catch (\Exception $e) {
                // Use as-is
            }

            Stripe::setApiKey($stripeSecret);

            // Build line items
            $lineItems = [];
            foreach ($order->items as $item) {
                $lineItems[] = [
                    'price_data' => [
                        'currency' => 'dop',
                        'product_data' => [
                            'name' => $item->product_name ?: 'Producto',
                        ],
                        'unit_amount' => (int) ($item->price * 100),
                    ],
                    'quantity' => $item->quantity,
                ];
            }

            // Add shipping/tax if needed
            if ($order->shipping > 0) {
                $lineItems[] = [
                    'price_data' => [
                        'currency' => 'dop',
                        'product_data' => ['name' => 'Envío'],
                        'unit_amount' => (int) ($order->shipping * 100),
                    ],
                    'quantity' => 1,
                ];
            }

            /** @var \App\Models\User $user */
            $user = auth()->user();
            $session = StripeSession::create([
                'payment_method_types' => ['card'],
                'line_items' => $lineItems,
                'mode' => 'payment',
                'success_url' => route('store.payment.stripe.success', ['order' => $order->id]) . '?session_id={CHECKOUT_SESSION_ID}',
                'cancel_url' => route('store.payment.stripe.cancel', ['order' => $order->id]),
                'metadata' => [
                    'order_id' => $order->id,
                    'order_number' => $order->order_number,
                ],
                'customer_email' => $user->email,
            ]);

            $order->update(['payment_id' => $session->id]);

            return Inertia::location($session->url);

        } catch (\Exception $e) {
            Log::error('Stripe error: ' . $e->getMessage());
            return redirect()->route('client.store.orders.show', $order->id)
                ->with('error', 'Error al procesar el pago: ' . $e->getMessage());
        }
    }

    /**
     * Stripe payment success callback.
     */
    public function stripeSuccess(Request $request, Order $order)
    {
        /** @var int|null $userId */
        $userId = auth()->id();
        if ($order->user_id !== $userId) {
            abort(403);
        }

        $sessionId = $request->query('session_id');

        try {
            $stripeSecret = Setting::get('stripe_secret');
            try {
                $stripeSecret = decrypt($stripeSecret);
            } catch (\Exception $e) {}

            Stripe::setApiKey($stripeSecret);

            $session = StripeSession::retrieve($sessionId);

            if ($session->payment_status === 'paid') {
                $order->update([
                    'status' => 'processing',
                    'payment_status' => 'paid',
                    'paid_at' => now(),
                    'transaction_id' => $session->payment_intent,
                ]);

                // Send payment confirmation to customer
                $order->user->notify(new OrderPaid($order));

                // Notify admins about payment received
                $admins = User::where('role', 'admin')->get();
                Notification::send($admins, new AdminStorePaymentReceived($order, $session->payment_intent));

                return redirect()->route('client.store.orders.show', $order->id)
                    ->with('success', '¡Pago completado! Tu pedido está siendo procesado.');
            }

            return redirect()->route('client.store.orders.show', $order->id)
                ->with('error', 'El pago no pudo ser verificado.');

        } catch (\Exception $e) {
            Log::error('Stripe verification error: ' . $e->getMessage());

            // Notify user and admins about failed payment
            $this->notifyPaymentFailed($order, $e->getMessage(), 'stripe');

            return redirect()->route('client.store.orders.show', $order->id)
                ->with('error', 'Error al verificar el pago.');
        }
    }

    /**
     * Stripe payment cancelled.
     */
    public function stripeCancel(Order $order)
    {
        return redirect()->route('client.store.orders.show', $order->id)
            ->with('warning', 'Pago cancelado. Puedes intentar de nuevo cuando quieras.');
    }

    /**
     * Initialize PayPal client.
     */
    protected function initPayPal(): PayPalClient
    {
        $paypalClientId = Setting::get('paypal_client_id');
        $paypalSecret = Setting::get('paypal_secret');
        $sandbox = Setting::get('paypal_sandbox', true);

        if (empty($paypalClientId) || empty($paypalSecret)) {
            throw new \Exception('PayPal no está configurado correctamente.');
        }

        try {
            $paypalSecret = decrypt($paypalSecret);
        } catch (\Exception $e) {
            // Use as-is
        }

        $config = [
            'mode' => $sandbox ? 'sandbox' : 'live',
            'sandbox' => [
                'client_id' => $paypalClientId,
                'client_secret' => $paypalSecret,
                'app_id' => '',
            ],
            'live' => [
                'client_id' => $paypalClientId,
                'client_secret' => $paypalSecret,
                'app_id' => '',
            ],
            'payment_action' => 'Sale',
            'currency' => 'DOP',
            'notify_url' => '',
            'locale' => 'es_ES',
            'validate_ssl' => !$sandbox,
        ];

        $paypal = new PayPalClient($config);
        $paypal->setApiCredentials($config);
        $paypal->getAccessToken();

        return $paypal;
    }

    /**
     * Send payment failed notifications to user and admins.
     */
    protected function notifyPaymentFailed(Order $order, ?string $reason = null, ?string $gateway = null): void
    {
        try {
            // Notify customer about failed payment
            $order->user->notify(new OrderPaymentFailed($order, $reason));

            // Notify all admins
            $admins = User::where('role', 'admin')->get();
            if ($admins->isNotEmpty()) {
                Notification::send($admins, new AdminOrderPaymentFailed($order, $reason, $gateway));
            }
        } catch (\Exception $e) {
            Log::error('Failed to send payment failed notifications', [
                'order_id' => $order->id,
                'error' => $e->getMessage(),
            ]);
        }
    }
}
