<?php

namespace App\Http\Controllers;

use App\Models\Plan;
use App\Models\ReferralCode;
use App\Models\Setting;
use App\Models\Subscription;
use App\Models\User;
use App\Notifications\SubscriptionActivated;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Inertia\Inertia;
use Stripe\Stripe;
use Stripe\Checkout\Session as StripeSession;

class PlanController extends Controller
{
    /**
     * Handle plan purchase intent for non-authenticated users.
     * Saves the plan to session and redirects to login/register.
     */
    public function intent(Plan $plan)
    {
        if (!$plan->is_active) {
            abort(404);
        }

        // Save the plan purchase intent in session
        session()->put('plan_purchase_intent', [
            'plan_id' => $plan->id,
            'plan_slug' => $plan->slug,
            'timestamp' => now()->timestamp,
        ]);

        // Set the intended URL to the checkout
        session()->put('url.intended', route('coach.subscription.show', ['plan' => $plan->slug]));

        // If user is already authenticated, redirect directly to checkout
        if (auth()->check()) {
            return redirect()->route('coach.subscription.show', ['plan' => $plan->slug]);
        }

        // Redirect to login with plan info
        return redirect()->route('login')
            ->with('info', 'Por favor, inicia sesión o regístrate para continuar con la compra del plan ' . $plan->name);
    }

    /**
     * Display a specific plan details page.
     */
    public function show(Plan $plan)
    {
        if (!$plan->is_active) {
            abort(404);
        }

        return Inertia::render('Plans/Show', [
            'plan' => [
                'id' => $plan->id,
                'name' => $plan->name,
                'slug' => $plan->slug,
                'description' => $plan->description,
                'price' => $plan->price,
                'monthly_price' => $plan->monthly_price,
                'quarterly_price' => $plan->quarterly_price,
                'yearly_price' => $plan->yearly_price,
                'duration_days' => $plan->duration_days,
                'features' => $plan->features,
            ],
            'stripeKey' => Setting::get('stripe_key'),
            'paypalClientId' => Setting::get('paypal_client_id'),
        ]);
    }

    /**
     * Create a Stripe checkout session for plan subscription.
     */
    public function createStripeSession(Request $request, Plan $plan)
    {
        $request->validate([
            'billing_cycle' => 'sometimes|string|in:monthly,quarterly,yearly',
        ]);

        $user = $request->user();
        $billingCycle = $request->billing_cycle ?? 'monthly';

        // Get the appropriate price
        $price = $plan->getPriceForCycle($billingCycle);

        // Get Stripe secret
        $stripeSecret = Setting::get('stripe_secret');
        if (!$stripeSecret) {
            return response()->json(['error' => 'Stripe no está configurado'], 500);
        }

        Stripe::setApiKey($stripeSecret);

        try {
            // Create Stripe checkout session
            $session = StripeSession::create([
                'payment_method_types' => ['card'],
                'line_items' => [[
                    'price_data' => [
                        'currency' => 'eur',
                        'product_data' => [
                            'name' => $plan->name,
                            'description' => $plan->description ?? "Plan {$plan->name} - {$billingCycle}",
                        ],
                        'unit_amount' => (int) ($price * 100), // Convert to cents
                    ],
                    'quantity' => 1,
                ]],
                'mode' => 'payment',
                'success_url' => route('plans.success') . '?session_id={CHECKOUT_SESSION_ID}',
                'cancel_url' => route('plans.cancel'),
                'client_reference_id' => $user->id,
                'metadata' => [
                    'user_id' => $user->id,
                    'plan_id' => $plan->id,
                    'billing_cycle' => $billingCycle,
                ],
                'customer_email' => $user->email,
            ]);

            return response()->json([
                'sessionId' => $session->id,
                'url' => $session->url,
            ]);
        } catch (\Exception $e) {
            Log::error('Stripe session creation failed', [
                'error' => $e->getMessage(),
                'user_id' => $user->id,
                'plan_id' => $plan->id,
            ]);

            return response()->json(['error' => 'Error al crear la sesión de pago'], 500);
        }
    }

    /**
     * Handle Stripe webhook for successful payments.
     */
    public function handleStripeWebhook(Request $request)
    {
        $payload = $request->getContent();
        $sigHeader = $request->header('Stripe-Signature');
        $webhookSecret = Setting::get('stripe_webhook_secret');

        try {
            $event = \Stripe\Webhook::constructEvent(
                $payload,
                $sigHeader,
                $webhookSecret
            );
        } catch (\UnexpectedValueException $e) {
            Log::error('Stripe webhook invalid payload');
            return response()->json(['error' => 'Invalid payload'], 400);
        } catch (\Stripe\Exception\SignatureVerificationException $e) {
            Log::error('Stripe webhook signature verification failed');
            return response()->json(['error' => 'Invalid signature'], 400);
        }

        // Handle the checkout.session.completed event
        if ($event->type === 'checkout.session.completed') {
            $session = $event->data->object;
            $this->handleSuccessfulPayment($session);
        }

        return response()->json(['status' => 'success']);
    }

    /**
     * Process successful payment and activate subscription.
     */
    protected function handleSuccessfulPayment($session)
    {
        $userId = $session->metadata->user_id ?? $session->client_reference_id;
        $planId = $session->metadata->plan_id;
        $billingCycle = $session->metadata->billing_cycle ?? 'monthly';

        $user = User::find($userId);
        $plan = Plan::find($planId);

        if (!$user || !$plan) {
            Log::error('Invalid user or plan in webhook', [
                'user_id' => $userId,
                'plan_id' => $planId,
            ]);
            return;
        }

        // Calculate end date based on billing cycle
        $endDate = match($billingCycle) {
            'monthly' => now()->addMonth(),
            'quarterly' => now()->addMonths(3),
            'yearly' => now()->addYear(),
            default => now()->addDays($plan->duration_days),
        };

        // Cancel any existing active subscription
        Subscription::where('user_id', $user->id)
            ->where('status', 'active')
            ->update(['status' => 'cancelled', 'end_date' => now()]);

        // Create new subscription
        $subscription = Subscription::create([
            'user_id' => $user->id,
            'plan_id' => $plan->id,
            'start_date' => now(),
            'end_date' => $endDate,
            'status' => 'active',
            'payment_method' => 'stripe',
            'stripe_session_id' => $session->id,
            'amount_paid' => $session->amount_total / 100,
        ]);

        // Update user's plan reference
        $user->update([
            'plan_id' => $plan->id,
            'subscription_status' => 'active',
        ]);

        // Notify user
        $user->notify(new SubscriptionActivated($subscription));

        Log::info('Subscription activated via Stripe', [
            'user_id' => $user->id,
            'plan_id' => $plan->id,
            'subscription_id' => $subscription->id,
        ]);
    }

    /**
     * Handle PayPal payment creation.
     */
    public function createPayPalOrder(Request $request, Plan $plan)
    {
        $request->validate([
            'billing_cycle' => 'sometimes|string|in:monthly,quarterly,yearly',
        ]);

        $user = $request->user();
        $billingCycle = $request->billing_cycle ?? 'monthly';
        $price = $plan->getPriceForCycle($billingCycle);

        $paypalClientId = Setting::get('paypal_client_id');
        $paypalSecret = Setting::get('paypal_secret');
        $sandbox = Setting::get('paypal_sandbox', true);

        if (!$paypalClientId || !$paypalSecret) {
            return response()->json(['error' => 'PayPal no está configurado'], 500);
        }

        $baseUrl = $sandbox 
            ? 'https://api-m.sandbox.paypal.com'
            : 'https://api-m.paypal.com';

        try {
            // Get access token
            $response = \Illuminate\Support\Facades\Http::withBasicAuth($paypalClientId, $paypalSecret)
                ->asForm()
                ->post("{$baseUrl}/v1/oauth2/token", [
                    'grant_type' => 'client_credentials',
                ]);

            $accessToken = $response->json('access_token');

            // Create order
            $orderResponse = \Illuminate\Support\Facades\Http::withToken($accessToken)
                ->post("{$baseUrl}/v2/checkout/orders", [
                    'intent' => 'CAPTURE',
                    'purchase_units' => [[
                        'reference_id' => "plan_{$plan->id}_user_{$user->id}",
                        'description' => "{$plan->name} - {$billingCycle}",
                        'amount' => [
                            'currency_code' => 'EUR',
                            'value' => number_format($price, 2, '.', ''),
                        ],
                        'custom_id' => json_encode([
                            'user_id' => $user->id,
                            'plan_id' => $plan->id,
                            'billing_cycle' => $billingCycle,
                        ]),
                    ]],
                    'application_context' => [
                        'return_url' => route('plans.paypal.success'),
                        'cancel_url' => route('plans.cancel'),
                    ],
                ]);

            return response()->json([
                'orderId' => $orderResponse->json('id'),
            ]);
        } catch (\Exception $e) {
            Log::error('PayPal order creation failed', [
                'error' => $e->getMessage(),
                'user_id' => $user->id,
                'plan_id' => $plan->id,
            ]);

            return response()->json(['error' => 'Error al crear el pedido de PayPal'], 500);
        }
    }

    /**
     * Capture PayPal payment.
     */
    public function capturePayPalOrder(Request $request)
    {
        $request->validate([
            'order_id' => 'required|string',
        ]);

        $orderId = $request->order_id;

        $paypalClientId = Setting::get('paypal_client_id');
        $paypalSecret = Setting::get('paypal_secret');
        $sandbox = Setting::get('paypal_sandbox', true);

        $baseUrl = $sandbox 
            ? 'https://api-m.sandbox.paypal.com'
            : 'https://api-m.paypal.com';

        try {
            // Get access token
            $response = \Illuminate\Support\Facades\Http::withBasicAuth($paypalClientId, $paypalSecret)
                ->asForm()
                ->post("{$baseUrl}/v1/oauth2/token", [
                    'grant_type' => 'client_credentials',
                ]);

            $accessToken = $response->json('access_token');

            // Capture order
            $captureResponse = \Illuminate\Support\Facades\Http::withToken($accessToken)
                ->post("{$baseUrl}/v2/checkout/orders/{$orderId}/capture");

            $captureData = $captureResponse->json();

            if ($captureData['status'] === 'COMPLETED') {
                $customId = json_decode($captureData['purchase_units'][0]['payments']['captures'][0]['custom_id'] ?? '{}', true);
                
                $this->activatePayPalSubscription(
                    $customId['user_id'],
                    $customId['plan_id'],
                    $customId['billing_cycle'] ?? 'monthly',
                    $orderId,
                    $captureData['purchase_units'][0]['payments']['captures'][0]['amount']['value']
                );

                return response()->json(['success' => true]);
            }

            return response()->json(['error' => 'Payment not completed'], 400);
        } catch (\Exception $e) {
            Log::error('PayPal capture failed', [
                'error' => $e->getMessage(),
                'order_id' => $orderId,
            ]);

            return response()->json(['error' => 'Error al capturar el pago'], 500);
        }
    }

    /**
     * Activate subscription after PayPal payment.
     */
    protected function activatePayPalSubscription($userId, $planId, $billingCycle, $paypalOrderId, $amountPaid)
    {
        $user = User::find($userId);
        $plan = Plan::find($planId);

        if (!$user || !$plan) {
            Log::error('Invalid user or plan for PayPal', [
                'user_id' => $userId,
                'plan_id' => $planId,
            ]);
            return;
        }

        $endDate = match($billingCycle) {
            'monthly' => now()->addMonth(),
            'quarterly' => now()->addMonths(3),
            'yearly' => now()->addYear(),
            default => now()->addDays($plan->duration_days),
        };

        // Cancel any existing active subscription
        Subscription::where('user_id', $user->id)
            ->where('status', 'active')
            ->update(['status' => 'cancelled', 'end_date' => now()]);

        // Create new subscription
        $subscription = Subscription::create([
            'user_id' => $user->id,
            'plan_id' => $plan->id,
            'start_date' => now(),
            'end_date' => $endDate,
            'status' => 'active',
            'payment_method' => 'paypal',
            'paypal_order_id' => $paypalOrderId,
            'amount_paid' => $amountPaid,
        ]);

        // Update user
        $user->update([
            'plan_id' => $plan->id,
            'subscription_status' => 'active',
        ]);

        // Notify user
        $user->notify(new SubscriptionActivated($subscription));

        Log::info('Subscription activated via PayPal', [
            'user_id' => $user->id,
            'plan_id' => $plan->id,
            'subscription_id' => $subscription->id,
        ]);
    }

    /**
     * Handle successful payment redirect.
     */
    public function success(Request $request)
    {
        return Inertia::render('Plans/Success', [
            'message' => '¡Tu suscripción ha sido activada exitosamente!',
        ]);
    }

    /**
     * Handle cancelled payment redirect.
     */
    public function cancel()
    {
        return Inertia::render('Plans/Cancel', [
            'message' => 'El proceso de pago ha sido cancelado.',
        ]);
    }

    /**
     * Validate a referral code.
     */
    public function validateReferralCode(Request $request)
    {
        $request->validate([
            'code' => 'required|string|max:20',
        ]);

        $code = strtoupper($request->code);

        // First check in referral_codes table
        $referralCode = ReferralCode::where('code', $code)
            ->where('is_active', true)
            ->first();

        if ($referralCode && $referralCode->isValid()) {
            return response()->json([
                'valid' => true,
                'type' => 'referral_code',
                'message' => 'Código de referido válido',
            ]);
        }

        // Fallback: check user's personal referral code
        $userWithCode = User::where('referral_code', $code)->first();
        
        if ($userWithCode) {
            return response()->json([
                'valid' => true,
                'type' => 'user_referral',
                'message' => 'Código de referido válido',
            ]);
        }

        return response()->json([
            'valid' => false,
            'message' => 'Código de referido no válido o expirado',
        ], 422);
    }
}
