<?php

namespace App\Http\Controllers\Coach;

use App\Http\Controllers\Controller;
use App\Models\Client;
use App\Models\User;
use App\Notifications\ClientPortalActivated;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

class ClientPortalController extends Controller
{
    /**
     * Activate client portal access.
     */
    public function activate(Request $request, Client $client): RedirectResponse
    {
        // 1. Check permissions
        $this->authorize('update', $client);

        if ($client->client_user_id) {
            return back()->with('error', 'Este cliente ya tiene acceso al portal.');
        }

        if (!$client->email) {
            return back()->with('error', 'El cliente debe tener un email para activar el portal.');
        }

        // Check if email already exists
        $existingUser = User::where('email', $client->email)->first();
        if ($existingUser) {
            return back()->with('error', 'Ya existe un usuario con este email.');
        }

        $coach = $request->user();
        $sendEmail = $request->boolean('send_email', true);

        // 2. Create User account
        $password = Str::random(10); // Temporary password

        $user = User::create([
            'name' => $client->name,
            'email' => $client->email,
            'password' => Hash::make($password),
            'role' => 'client',
            'status' => 'active',
            'phone' => $client->phone,
            'avatar' => $client->avatar,
            'email_verified_at' => now(),
        ]);

        // 3. Link Client to User and save password for coach viewing
        $client->update([
            'client_user_id' => $user->id,
            'portal_active' => true,
            'portal_password' => $password,
        ]);

        // 4. Send email with credentials
        if ($sendEmail) {
            try {
                $user->notify(new ClientPortalActivated($client, $password, $coach->name));
                return back()->with('success', 'Portal activado. Se han enviado las credenciales por email.');
            } catch (\Exception $e) {
                return back()->with('success', "Portal activado. Credenciales: Email: {$client->email}, Contraseña: {$password}");
            }
        }

        return back()->with('success', "Portal activado. Credenciales: Email: {$client->email}, Contraseña: {$password}");
    }

    /**
     * Deactivate client portal access.
     */
    public function deactivate(Client $client): RedirectResponse
    {
        $this->authorize('update', $client);

        if ($client->client_user_id) {
            $user = User::find($client->client_user_id);
            $client->update([
                'client_user_id' => null,
                'portal_active' => false,
            ]);
            $user?->delete();
        }

        return back()->with('success', 'Acceso al portal desactivado.');
    }

    /**
     * Resend credentials to client.
     */
    public function resendCredentials(Request $request, Client $client): RedirectResponse
    {
        $this->authorize('update', $client);

        if (!$client->client_user_id) {
            return back()->with('error', 'El cliente no tiene portal activado.');
        }

        $user = User::find($client->client_user_id);
        if (!$user) {
            return back()->with('error', 'Usuario no encontrado.');
        }

        $coach = $request->user();
        
        // Generate new password and save for coach viewing
        $password = Str::random(10);
        $user->update(['password' => Hash::make($password)]);
        $client->update(['portal_password' => $password]);

        // Send email
        try {
            $user->notify(new ClientPortalActivated($client, $password, $coach->name));
            return back()->with('success', 'Nuevas credenciales enviadas por email.');
        } catch (\Exception $e) {
            return back()->with('success', "Nueva contraseña: {$password}");
        }
    }
}
