<?php

namespace App\Http\Controllers\Client;

use App\Http\Controllers\Controller;
use App\Models\ComplianceLog;
use App\Models\ProgressPhoto;
use Carbon\Carbon;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Inertia\Inertia;

class ProgressController extends Controller
{
    /**
     * Display the client's progress page with metrics and charts
     */
    public function index()
    {
        $user = auth()->user();
        $client = $user->clientProfile ?? $user->client;
        
        if (!$client) {
            return Inertia::render('Client/Progress/Index', [
                'client' => null,
                'photos' => [],
                'compliance_history' => [],
                'weight_history' => [],
                'stats' => null,
                'no_profile' => true,
            ]);
        }
        
        // Get progress photos with weight data for chart
        $photos = $client->progressPhotos()
            ->orderBy('date', 'desc')
            ->get();

        // Build weight history from photos (they contain weight data)
        $weightHistory = $photos->map(fn($photo) => [
            'date' => Carbon::parse($photo->date)->format('d M'),
            'weight' => $photo->weight,
            'raw_date' => $photo->date,
        ])->sortBy('raw_date')->values()->take(30);

        // Get compliance history for the last 30 days
        $complianceHistory = [];
        for ($i = 29; $i >= 0; $i--) {
            $date = Carbon::today()->subDays($i);
            $workouts = $client->complianceLogs()
                ->where('resource_type', 'workout')
                ->whereDate('completed_at', $date)
                ->count();
            $meals = $client->complianceLogs()
                ->where('resource_type', 'meal')
                ->whereDate('completed_at', $date)
                ->count();
            
            $complianceHistory[] = [
                'date' => $date->format('d/m'),
                'day' => $date->format('D'),
                'workouts' => $workouts,
                'meals' => $meals,
            ];
        }

        // Calculate stats
        $totalWorkouts = $client->complianceLogs()->where('resource_type', 'workout')->count();
        $totalMeals = $client->complianceLogs()->where('resource_type', 'meal')->count();
        
        // Weight change calculation
        $startWeight = $photos->sortBy('date')->first()?->weight ?? $client->weight;
        $currentWeight = $photos->sortByDesc('date')->first()?->weight ?? $client->weight;
        
        $stats = [
            'current_weight' => $currentWeight ?? $client->weight,
            'start_weight' => $startWeight ?? $client->weight,
            'weight_change' => $startWeight && $currentWeight ? round($currentWeight - $startWeight, 1) : 0,
            'bmi' => $client->bmi ? round($client->bmi, 1) : null,
            'tdee' => $client->tdee ? round($client->tdee) : null,
            'bmr' => $client->bmr ? round($client->bmr) : null,
            'total_workouts' => $totalWorkouts,
            'total_meals' => $totalMeals,
            'total_photos' => $photos->count(),
        ];
        
        return Inertia::render('Client/Progress/Index', [
            'client' => $client,
            'photos' => $photos,
            'compliance_history' => $complianceHistory,
            'weight_history' => $weightHistory,
            'stats' => $stats,
        ]);
    }

    /**
     * Log compliance for a workout or meal.
     */
    public function logCompliance(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'resource_type' => 'required|in:workout,meal',
            'resource_id' => 'required|integer',
        ]);

        $client = $request->user()->clientProfile;

        if (!$client) {
            return back()->with('error', 'No tienes un perfil de cliente asociado.');
        }

        ComplianceLog::create([
            'client_id' => $client->id,
            'resource_type' => $validated['resource_type'],
            'resource_id' => $validated['resource_id'],
            'completed_at' => now(),
        ]);

        return back()->with('success', '¡Excelente trabajo! Se ha registrado tu cumplimiento.');
    }

    /**
     * Upload a progress photo.
     */
    public function uploadPhoto(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'image' => 'required|image|max:10240', // 10MB max
            'type' => 'required|in:front,side,back',
            'weight' => 'nullable|numeric|min:30|max:300',
            'date' => 'required|date',
        ]);

        $client = $request->user()->clientProfile;

        if (!$client) {
            return back()->with('error', 'No tienes un perfil de cliente asociado.');
        }

        if ($request->hasFile('image')) {
            $path = $request->file('image')->store('progress_photos/' . $client->id, 'public');

            ProgressPhoto::create([
                'client_id' => $client->id,
                'image_path' => '/storage/' . $path,
                'type' => $validated['type'],
                'weight' => $validated['weight'] ?? $client->weight,
                'date' => $validated['date'],
            ]);
        }

        return back()->with('success', 'Foto de progreso subida correctamente.');
    }

    /**
     * Delete a progress photo.
     */
    public function deletePhoto(ProgressPhoto $photo): RedirectResponse
    {
        // Ensure user owns the client profile that owns the photo
        if (auth()->user()->clientProfile->id !== $photo->client_id) {
            abort(403);
        }

        // Delete from storage
        $path = str_replace('/storage/', '', $photo->image_path);
        Storage::disk('public')->delete($path);

        $photo->delete();

        return back()->with('success', 'Foto eliminada.');
    }
}
