<?php

namespace App\Http\Controllers\Client;

use App\Http\Controllers\Controller;
use App\Models\MealPlan;
use App\Services\PdfService;
use App\Services\PdfBrowsershotService;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Inertia\Inertia;

/**
 * MealPlanController
 * Handles client meal plan views
 */
class MealPlanController extends Controller
{
    /**
     * Display the client's meal plans
     */
    public function index()
    {
        $user = auth()->user();
        $client = $user->clientProfile ?? $user->client;
        
        if (!$client) {
            return Inertia::render('Client/MealPlan/Index', [
                'client' => null,
                'mealPlans' => [],
                'compliance_logs' => [],
                'completion_counts' => [],
                'no_profile' => true,
            ]);
        }
        
        $mealPlans = $client->mealPlans()
            ->where('is_active', true)
            ->with(['dietPlans.items.food'])
            ->get();
            
        $complianceLogs = $client->complianceLogs()
            ->where('resource_type', 'meal')
            ->whereDate('completed_at', '>=', now()->subDays(30))
            ->get();

        // Calculate completion counts per meal plan
        $completionCounts = $complianceLogs->groupBy('resource_id')
            ->map(fn($logs) => $logs->count());
        
        return Inertia::render('Client/MealPlan/Index', [
            'client' => $client,
            'mealPlans' => $mealPlans,
            'compliance_logs' => $complianceLogs,
            'completion_counts' => $completionCounts,
        ]);
    }
    
    /**
     * Generate and download PDF for a meal plan.
     * Uses the same PdfBrowsershotService as the coach for consistent design.
     */
    public function pdf($mealPlanId)
    {
        $user = auth()->user();
        $client = $user->clientProfile ?? $user->client;
        
        if (!$client) {
            abort(404, 'Cliente no encontrado');
        }
        
        $mealPlan = $client->mealPlans()->findOrFail($mealPlanId);
        
        // Get language from query parameter (default to Spanish)
        $lang = request()->query('lang', 'es');
        
        try {
            // Use PdfBrowsershotService (same as coach) for consistent PDF design
            $browsershotService = app(PdfBrowsershotService::class);
            
            try {
                $filename = $browsershotService->generateMealPlan($mealPlan, $lang);
            } catch (\Exception $e) {
                // Fallback to DomPDF if Browsershot fails
                Log::warning('Client Browsershot failed, using DomPDF: ' . $e->getMessage());
                $pdfService = app(PdfService::class);
                $filename = $pdfService->generateMealPlan($mealPlan, $lang);
            }
            
            // Use public disk to get the file
            $disk = Storage::disk('public');
            
            if (!$disk->exists($filename)) {
                return redirect()->back()->with('error', 'No se pudo generar el PDF.');
            }
            
            // File name based on language
            $langSuffix = $lang === 'en' ? '-EN' : '-ES';
            
            return response()->download(
                $disk->path($filename),
                "dieta-{$client->name}-{$mealPlan->name}{$langSuffix}.pdf"
            );
            
        } catch (\Exception $e) {
            Log::error('Client PDF Error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Error al generar el PDF: ' . $e->getMessage());
        }
    }
    
    /**
     * Email the meal plan PDF to the client.
     */
    public function emailMealPlan($mealPlanId)
    {
        $user = auth()->user();
        $client = $user->clientProfile ?? $user->client;
        
        if (!$client) {
            return redirect()->back()->with('error', 'Cliente no encontrado');
        }
        
        $mealPlan = $client->mealPlans()->findOrFail($mealPlanId);
        
        // Get language from query parameter (default to Spanish)
        $lang = request()->query('lang', 'es');
        
        try {
            // Generate PDF using PdfBrowsershotService
            $browsershotService = app(PdfBrowsershotService::class);
            
            try {
                $filename = $browsershotService->generateMealPlan($mealPlan, $lang);
            } catch (\Exception $e) {
                Log::warning('Client Browsershot failed for email, using DomPDF: ' . $e->getMessage());
                $pdfService = app(PdfService::class);
                $filename = $pdfService->generateMealPlan($mealPlan, $lang);
            }
            
            $disk = Storage::disk('public');
            
            if (!$disk->exists($filename)) {
                return redirect()->back()->with('error', 'No se pudo generar el PDF.');
            }
            
            $pdfPath = $disk->path($filename);
            $langSuffix = $lang === 'en' ? '-EN' : '-ES';
            $attachmentName = "dieta-{$client->name}-{$mealPlan->name}{$langSuffix}.pdf";
            
            // Send email
            \Mail::to($user->email)->send(new \App\Mail\MealPlanEmail(
                $client,
                $mealPlan,
                $pdfPath,
                $attachmentName
            ));
            
            return redirect()->back()->with('success', '✅ Plan de dieta enviado a tu correo: ' . $user->email);
            
        } catch (\Exception $e) {
            Log::error('Client Email Error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Error al enviar el correo: ' . $e->getMessage());
        }
    }
}
