<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\OAuthSetting;
use App\Models\User;
use Exception;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Laravel\Socialite\Facades\Socialite;

class GoogleController extends Controller
{
    /**
     * Redirect the user to Google authentication page.
     */
    public function redirect(): RedirectResponse
    {
        // Check if Google OAuth is enabled
        if (!OAuthSetting::isProviderEnabled('google')) {
            return redirect()->route('login')
                ->with('error', 'El inicio de sesión con Google no está habilitado.');
        }

        // Configure Google OAuth dynamically
        $this->configureGoogleProvider();

        return Socialite::driver('google')
            ->scopes(['openid', 'profile', 'email'])
            ->redirect();
    }

    /**
     * Handle the callback from Google.
     */
    public function callback(): RedirectResponse
    {
        // Check if Google OAuth is enabled
        if (!OAuthSetting::isProviderEnabled('google')) {
            return redirect()->route('login')
                ->with('error', 'El inicio de sesión con Google no está habilitado.');
        }

        try {
            // Configure Google OAuth dynamically
            $this->configureGoogleProvider();

            $googleUser = Socialite::driver('google')->user();

            // Check if user exists by Google ID
            $user = User::where('google_id', $googleUser->getId())->first();

            if ($user) {
                // Update Google tokens
                $user->update([
                    'google_token' => $googleUser->token,
                    'google_refresh_token' => $googleUser->refreshToken,
                    'google_avatar' => $googleUser->getAvatar(),
                ]);

                // Check user status
                if ($user->status === 'inactive') {
                    return redirect()->route('login')
                        ->with('error', 'Tu cuenta está desactivada. Contacta al administrador.');
                }

                if ($user->status === 'pending') {
                    Auth::login($user);
                    return redirect()->route('pending.approval');
                }

                Auth::login($user, true);
                return $this->redirectBasedOnRole($user);
            }

            // Check if user exists by email
            $existingUser = User::where('email', $googleUser->getEmail())->first();

            if ($existingUser) {
                // Link Google account to existing user
                $existingUser->update([
                    'google_id' => $googleUser->getId(),
                    'google_token' => $googleUser->token,
                    'google_refresh_token' => $googleUser->refreshToken,
                    'google_avatar' => $googleUser->getAvatar(),
                ]);

                // Check user status
                if ($existingUser->status === 'inactive') {
                    return redirect()->route('login')
                        ->with('error', 'Tu cuenta está desactivada. Contacta al administrador.');
                }

                if ($existingUser->status === 'pending') {
                    Auth::login($existingUser);
                    return redirect()->route('pending.approval');
                }

                Auth::login($existingUser, true);
                return $this->redirectBasedOnRole($existingUser);
            }

            // Create new user as coach (default role for Google sign-up)
            $user = User::create([
                'name' => $googleUser->getName(),
                'email' => $googleUser->getEmail(),
                'password' => Hash::make(Str::random(24)),
                'role' => 'coach',
                'status' => 'pending', // New coaches need approval
                'google_id' => $googleUser->getId(),
                'google_token' => $googleUser->token,
                'google_refresh_token' => $googleUser->refreshToken,
                'google_avatar' => $googleUser->getAvatar(),
                'avatar' => $googleUser->getAvatar(),
                'email_verified_at' => now(), // Google accounts are verified
                'referral_code' => User::generateReferralCode(),
            ]);

            Auth::login($user);

            // Redirect to pending approval page
            return redirect()->route('pending.approval')
                ->with('success', '¡Cuenta creada exitosamente! Tu cuenta está pendiente de aprobación.');

        } catch (Exception $e) {
            Log::error('Google OAuth Error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
            ]);

            return redirect()->route('login')
                ->with('error', 'Error al iniciar sesión con Google. Por favor, intenta de nuevo.');
        }
    }

    /**
     * Configure Google provider dynamically from database settings.
     */
    private function configureGoogleProvider(): void
    {
        $config = OAuthSetting::getGoogleConfig();

        config([
            'services.google.client_id' => $config['client_id'] ?? '',
            'services.google.client_secret' => $config['client_secret'] ?? '',
            'services.google.redirect' => $config['redirect'] ?? route('auth.google.callback'),
        ]);
    }

    /**
     * Redirect user based on their role.
     */
    private function redirectBasedOnRole(User $user): RedirectResponse
    {
        return match ($user->role) {
            'admin' => redirect()->intended(route('admin.dashboard')),
            'coach' => redirect()->intended(route('coach.dashboard')),
            'client' => redirect()->intended(route('client.dashboard')),
            default => redirect()->intended(route('dashboard')),
        };
    }
}
