<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\ProductCategory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Inertia\Inertia;

class ProductController extends Controller
{
    public function index(Request $request)
    {
        $query = Product::with('category')
            ->when($request->search, function ($q, $search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('sku', 'like', "%{$search}%");
            })
            ->when($request->category, function ($q, $category) {
                $q->where('category_id', $category);
            })
            ->when($request->status !== null, function ($q) use ($request) {
                $q->where('is_active', $request->status === 'active');
            })
            ->latest();

        return Inertia::render('Admin/Store/Products/Index', [
            'products' => $query->paginate(15)->withQueryString(),
            'categories' => ProductCategory::active()->ordered()->get(),
            'filters' => $request->only(['search', 'category', 'status']),
        ]);
    }

    public function create()
    {
        return Inertia::render('Admin/Store/Products/Create', [
            'categories' => ProductCategory::active()->ordered()->get(),
        ]);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'short_description' => 'nullable|string|max:500',
            'sku' => 'nullable|string|unique:products,sku',
            'price' => 'required|numeric|min:0',
            'compare_price' => 'nullable|numeric|min:0',
            'price_dop' => 'nullable|numeric|min:0',
            'compare_price_dop' => 'nullable|numeric|min:0',
            'cost' => 'nullable|numeric|min:0',
            'stock' => 'required|integer|min:0',
            'track_stock' => 'boolean',
            'allow_backorder' => 'boolean',
            'category_id' => 'nullable|exists:product_categories,id',
            'weight' => 'nullable|numeric|min:0',
            'dimensions' => 'nullable|string|max:100',
            'tax_rate' => 'numeric|min:0|max:100',
            'is_active' => 'boolean',
            'is_featured' => 'boolean',
            'image' => 'nullable|image|max:5120',
            'gallery' => 'nullable|array',
            'gallery.*' => 'image|max:5120',
            'sizes' => 'nullable|array',
            'sizes.*.name' => 'required_with:sizes|string|max:50',
            'sizes.*.price' => 'required_with:sizes|numeric|min:0',
            'flavors' => 'nullable|array',
            'flavors.*' => 'string|max:100',
            'nutrition_info' => 'nullable|array',
        ]);

        $validated['slug'] = Str::slug($validated['name']);

        if ($request->hasFile('image')) {
            $validated['image'] = $request->file('image')->store('products', 'public');
        }

        if ($request->hasFile('gallery')) {
            $gallery = [];
            foreach ($request->file('gallery') as $file) {
                $gallery[] = $file->store('products', 'public');
            }
            $validated['gallery'] = $gallery;
        }

        Product::create($validated);

        return redirect()->route('admin.store.products.index')
            ->with('success', 'Producto creado correctamente.');
    }

    public function edit(Product $product)
    {
        return Inertia::render('Admin/Store/Products/Edit', [
            'product' => $product,
            'categories' => ProductCategory::active()->ordered()->get(),
        ]);
    }

    public function update(Request $request, Product $product)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'short_description' => 'nullable|string|max:500',
            'sku' => 'nullable|string|unique:products,sku,' . $product->id,
            'price' => 'required|numeric|min:0',
            'compare_price' => 'nullable|numeric|min:0',
            'price_dop' => 'nullable|numeric|min:0',
            'compare_price_dop' => 'nullable|numeric|min:0',
            'cost' => 'nullable|numeric|min:0',
            'stock' => 'required|integer|min:0',
            'track_stock' => 'boolean',
            'allow_backorder' => 'boolean',
            'category_id' => 'nullable|exists:product_categories,id',
            'weight' => 'nullable|numeric|min:0',
            'dimensions' => 'nullable|string|max:100',
            'tax_rate' => 'numeric|min:0|max:100',
            'is_active' => 'boolean',
            'is_featured' => 'boolean',
            'image' => 'nullable|image|max:5120',
            'gallery' => 'nullable|array',
            'gallery.*' => 'image|max:5120',
            'sizes' => 'nullable|array',
            'sizes.*.name' => 'required_with:sizes|string|max:50',
            'sizes.*.price' => 'required_with:sizes|numeric|min:0',
            'flavors' => 'nullable|array',
            'flavors.*' => 'string|max:100',
            'nutrition_info' => 'nullable|array',
        ]);

        $validated['slug'] = Str::slug($validated['name']);

        if ($request->hasFile('image')) {
            // Delete old image
            if ($product->image && Storage::disk('public')->exists($product->image)) {
                Storage::disk('public')->delete($product->image);
            }
            $validated['image'] = $request->file('image')->store('products', 'public');
        }

        if ($request->hasFile('gallery')) {
            // Delete old gallery
            if ($product->gallery) {
                foreach ($product->gallery as $oldImage) {
                    if (Storage::disk('public')->exists($oldImage)) {
                        Storage::disk('public')->delete($oldImage);
                    }
                }
            }
            $gallery = [];
            foreach ($request->file('gallery') as $file) {
                $gallery[] = $file->store('products', 'public');
            }
            $validated['gallery'] = $gallery;
        }

        $product->update($validated);

        return redirect()->route('admin.store.products.index')
            ->with('success', 'Producto actualizado correctamente.');
    }

    public function destroy(Product $product)
    {
        // Delete images
        if ($product->image && Storage::disk('public')->exists($product->image)) {
            Storage::disk('public')->delete($product->image);
        }
        if ($product->gallery) {
            foreach ($product->gallery as $img) {
                if (Storage::disk('public')->exists($img)) {
                    Storage::disk('public')->delete($img);
                }
            }
        }

        $product->delete();

        return redirect()->route('admin.store.products.index')
            ->with('success', 'Producto eliminado correctamente.');
    }

    public function toggleStatus(Product $product)
    {
        $product->update(['is_active' => !$product->is_active]);

        return back()->with('success', 
            $product->is_active ? 'Producto activado.' : 'Producto desactivado.'
        );
    }

    public function toggleFeatured(Product $product)
    {
        $product->update(['is_featured' => !$product->is_featured]);

        return back()->with('success', 
            $product->is_featured ? 'Producto destacado.' : 'Producto removido de destacados.'
        );
    }
}
