<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Setting;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Inertia\Inertia;
use Inertia\Response;

class PageController extends Controller
{
    /**
     * Available static pages (contact removed - has its own dedicated page).
     */
    protected array $pages = [
        'about' => 'Quiénes Somos',
        'terms' => 'Términos y Condiciones',
        'privacy' => 'Política de Privacidad',
        'cookies' => 'Política de Cookies',
        'refund' => 'Política de Reembolso',
        'faq' => 'Preguntas Frecuentes',
    ];

    /**
     * Display a listing of static pages.
     */
    public function index(): Response
    {
        $pages = [];
        foreach ($this->pages as $slug => $name) {
            $pages[$slug] = [
                'name' => $name,
                'slug' => $slug,
                'content' => Setting::get("page_{$slug}", ''),
                'updated_at' => Setting::where('key', "page_{$slug}")->value('updated_at'),
            ];
        }

        return Inertia::render('Admin/Pages/Index', [
            'pages' => $pages,
        ]);
    }

    /**
     * Show the form for editing the specified page.
     */
    public function edit(string $slug): Response
    {
        if (!isset($this->pages[$slug])) {
            abort(404);
        }

        return Inertia::render('Admin/Pages/Edit', [
            'slug' => $slug,
            'name' => $this->pages[$slug],
            'content' => Setting::get("page_{$slug}", $this->getDefaultContent($slug)),
        ]);
    }

    /**
     * Update the specified page.
     */
    public function update(Request $request, string $slug): RedirectResponse
    {
        if (!isset($this->pages[$slug])) {
            abort(404);
        }

        $validated = $request->validate([
            'content' => 'required|string|max:50000',
        ]);

        Setting::set("page_{$slug}", $validated['content'], 'pages');

        return back()->with('success', 'Página actualizada correctamente.');
    }

    private function getDefaultContent(string $slug): string
    {
        $defaults = [
            'about' => '<h1>Sobre CoachPro</h1><p>CoachPro es la plataforma líder para entrenadores personales.</p>',
            'terms' => '<h1>Términos y Condiciones</h1><p>Al usar CoachPro aceptas estos términos...</p>',
            'privacy' => '<h1>Política de Privacidad</h1><p>Tu privacidad es importante para nosotros...</p>',
            'cookies' => '<h1>Política de Cookies</h1><p>Este sitio utiliza cookies para mejorar tu experiencia...</p>',
            'refund' => '<h1>Política de Reembolso</h1><p>Puedes solicitar reembolso dentro de los primeros 7 días...</p>',
            'faq' => '<h1>Preguntas Frecuentes</h1><p><strong>¿Cómo empiezo?</strong></p><p>Regístrate y comienza tu prueba gratis.</p>',
        ];

        return $defaults[$slug] ?? '<p>Contenido de la página...</p>';
    }
}
